import { SagaIterator } from 'redux-saga';
import { environmentForm, initialState } from './environment-form.reducer';
import environmentFormSaga, { cancel, initEnvironmentForm, save, toastr } from './environment-form.saga';
import { addConfirmLeavePopup, removeConfirmLeavePopup } from '@xlr-ui/app/react/utils/sagas';
import { all, call, put, takeLatest } from 'redux-saga/effects';
import { env, envInFolder, envList } from '../__mocks__/environment-list.mocks';
import getAngularService from '@xlr-ui/app/features/common/services/angular-accessor';
import map from 'lodash/map';
import { httpGET, httpPOST, httpPUT } from '@xlr-ui/app/features/common/services/http';
import { stageState } from './stage.reducer';
import { labelsState } from './labels.reducer';
import { environmentLists } from './environment-list.reducer';

const { cancelForm, initForm, pristineFormChanged, saveForm, setEnvironmentModel } = environmentForm.actions;
const { setEnvironments } = environmentLists.actions;

const { searchStages } = stageState.actions;
const { searchLabels } = labelsState.actions;

describe('environment-form.saga', () => {
    it('should yield all effects', () => {
        const gen: SagaIterator = environmentFormSaga();
        expect(gen.next().value).toStrictEqual(
            all([
                takeLatest(cancelForm, cancel),
                takeLatest(saveForm, save),
                takeLatest(initForm, initEnvironmentForm),
                takeLatest(pristineFormChanged, addConfirmLeavePopup),
            ]),
        );
        expect(gen.next().done).toBe(true);
    });

    describe('createEnvironmentModel', () => {
        it('should create app without folder', () => {
            const $state = { go: jest.fn() };
            const environmentFormFields = {
                title: env.title,
                stageId: env.stage.id,
                description: env.description,
                labelIds: map(env.labels, 'id'),
                folderId: env.folderId,
            };
            const gen: SagaIterator = save({
                payload: env,
                type: 'whatever',
            });
            expect(gen.next().value).toStrictEqual(call(getAngularService, '$state'));
            expect(gen.next($state).value).toStrictEqual(call(removeConfirmLeavePopup));
            expect(gen.next().value).toStrictEqual(call(httpPOST, 'api/v1/environments', environmentFormFields));
            expect(gen.next().value).toStrictEqual(call(toastr.success, `Environment [${env.title}] created`));
            expect(gen.next().value).toStrictEqual($state.go('environmentList'));
            expect(gen.next().done).toBe(true);
        });

        it('should create app in folder', () => {
            const $state = { go: jest.fn() };
            const environmentFormFields = {
                title: envInFolder.title,
                stageId: envInFolder.stage.id,
                description: envInFolder.description,
                labelIds: map(envInFolder.labels, 'id'),
                folderId: envInFolder.folderId,
            };
            const gen: SagaIterator = save({
                payload: envInFolder,
                type: 'whatever',
            });
            expect(gen.next().value).toStrictEqual(call(getAngularService, '$state'));
            expect(gen.next($state).value).toStrictEqual(call(removeConfirmLeavePopup));
            expect(gen.next().value).toStrictEqual(call(httpPOST, 'api/v1/environments', environmentFormFields));
            expect(gen.next().value).toStrictEqual(call(toastr.success, `Environment [${envInFolder.title}] created`));
            expect(gen.next().value).toStrictEqual($state.go('folders.detail.folderEnvironment'));
            expect(gen.next().done).toBe(true);
        });
    });

    describe('updateEnvironmentModel', () => {
        it('should update app without folder', () => {
            const $state = { go: jest.fn() };
            const updatedEnv = {
                ...env,
                id: '2',
                title: 'updated',
            };
            const environmentFormFields = {
                title: updatedEnv.title,
                stageId: env.stage.id,
                description: env.description,
                labelIds: map(env.labels, 'id'),
                folderId: env.folderId,
            };
            const gen: SagaIterator = save({
                payload: updatedEnv,
                type: 'whatever',
            });
            expect(gen.next().value).toStrictEqual(call(getAngularService, '$state'));
            expect(gen.next($state).value).toStrictEqual(call(removeConfirmLeavePopup));
            expect(gen.next().value).toStrictEqual(call(httpPUT, `api/v1/environments/${updatedEnv.id}`, environmentFormFields));
            expect(gen.next().value).toStrictEqual(call(toastr.success, `Environment [${updatedEnv.title}] updated`));
            expect(gen.next().value).toStrictEqual($state.go('environmentList'));
            expect(gen.next().done).toBe(true);
        });

        it('should update app in folder', () => {
            const $state = { go: jest.fn() };
            const updatedEnv = {
                ...envInFolder,
                id: '3',
                title: 'updated',
            };
            const environmentFormFields = {
                title: updatedEnv.title,
                stageId: envInFolder.stage.id,
                description: envInFolder.description,
                labelIds: map(envInFolder.labels, 'id'),
                folderId: envInFolder.folderId,
            };
            const gen: SagaIterator = save({
                payload: updatedEnv,
                type: 'whatever',
            });
            expect(gen.next().value).toStrictEqual(call(getAngularService, '$state'));
            expect(gen.next($state).value).toStrictEqual(call(removeConfirmLeavePopup));
            expect(gen.next().value).toStrictEqual(call(httpPUT, `api/v1/environments/${updatedEnv.id}`, environmentFormFields));
            expect(gen.next().value).toStrictEqual(call(toastr.success, `Environment [${updatedEnv.title}] updated`));
            expect(gen.next().value).toStrictEqual($state.go('folders.detail.folderEnvironment'));
            expect(gen.next().done).toBe(true);
        });
    });

    describe('cancel form', () => {
        it('should cancel form without folder', () => {
            const $state = { go: jest.fn() };
            const gen: SagaIterator = cancel({
                payload: '',
                type: 'whatever',
            });
            expect(gen.next().value).toStrictEqual(call(removeConfirmLeavePopup));
            expect(gen.next().value).toStrictEqual(call(getAngularService, '$state'));
            expect(gen.next($state).value).toStrictEqual($state.go('environmentList'));
            expect(gen.next().done).toBe(true);
        });

        it('should cancel form in folder', () => {
            const $state = { go: jest.fn() };
            const gen: SagaIterator = cancel({
                payload: 'Folder42',
                type: 'whatever',
            });
            expect(gen.next().value).toStrictEqual(call(removeConfirmLeavePopup));
            expect(gen.next().value).toStrictEqual(call(getAngularService, '$state'));
            expect(gen.next($state).value).toStrictEqual($state.go('folders.detail.folderEnvironment'));
            expect(gen.next().done).toBe(true);
        });
    });

    describe('init form', () => {
        it('init create form', () => {
            const gen: SagaIterator = initEnvironmentForm({
                payload: {
                    envId: '',
                    folderId: '',
                },
                type: 'whatever',
            });
            expect(gen.next().value).toStrictEqual(call(httpPOST, 'api/v1/environments/search', { folderId: '' }));
            expect(gen.next({ data: envList }).value).toStrictEqual(put(setEnvironments(envList)));
            expect(gen.next().value).toStrictEqual(put(setEnvironmentModel(initialState.environmentModel)));
            expect(gen.next().value).toStrictEqual(put(searchStages('')));
            expect(gen.next().value).toStrictEqual(put(searchLabels('')));
            expect(gen.next().done).toBe(true);
        });

        it('init update form', () => {
            const gen: SagaIterator = initEnvironmentForm({
                payload: {
                    envId: '2',
                    folderId: '',
                },
                type: 'whatever',
            });
            expect(gen.next().value).toStrictEqual(call(httpPOST, 'api/v1/environments/search', { folderId: '' }));
            expect(gen.next({ data: envList }).value).toStrictEqual(call(httpGET, `api/v1/environments/2`));
            expect(gen.next({ data: { ...env, id: '2' } }).value).toStrictEqual(put(setEnvironments(envList)));
            expect(gen.next().value).toStrictEqual(put(setEnvironmentModel({ ...env, id: '2' })));
            expect(gen.next().value).toStrictEqual(put(searchStages('')));
            expect(gen.next().value).toStrictEqual(put(searchLabels('')));
            expect(gen.next().done).toBe(true);
        });
    });
});
