package com.xebialabs.xlrelease.environments.service

import com.xebialabs.deployit.checks.Checks.{IncorrectArgumentException, checkArgument}
import com.xebialabs.xlrelease.api.v1.filter.ReservationFilters
import com.xebialabs.xlrelease.domain.environments.{Environment, EnvironmentReservation}
import com.xebialabs.xlrelease.environments.events.{EnvironmentReservationCreatedEvent, EnvironmentReservationDeletedEvent, EnvironmentReservationUpdatedEvent}
import com.xebialabs.xlrelease.environments.repository.EnvironmentReservationRepository
import com.xebialabs.xlrelease.events.XLReleaseEventBus
import grizzled.slf4j.Logging
import io.micrometer.core.annotation.Timed
import org.springframework.beans.factory.annotation.Autowired
import org.springframework.stereotype.Service
import org.springframework.util.CollectionUtils
import org.springframework.util.StringUtils.hasText

import java.util.{Date, List => JList, Map => JMap}
import scala.jdk.CollectionConverters._

@Service
class EnvironmentReservationService @Autowired()(environmentReservationRepository: EnvironmentReservationRepository,
                                                 applicationService: ApplicationService,
                                                 eventBus: XLReleaseEventBus) extends Logging {

  @Timed
  def searchReservations(filters: ReservationFilters): JMap[Environment, JList[EnvironmentReservation]] = {
    environmentReservationRepository.search(filters, defaultPage).view.mapValues(_.asJava).toMap.asJava
  }

  @Timed
  def findReservationById(environmentReservationId: String): EnvironmentReservation = {
    environmentReservationRepository.findById(environmentReservationId)
  }

  @Timed
  def getReservationsByEnvironmentId(environmentId: String): JList[EnvironmentReservation] = {
    environmentReservationRepository.getAllForEnvironmentId(environmentId).asJava
  }

  @Timed
  def createReservation(environmentReservation: EnvironmentReservation): EnvironmentReservation = {
    logger.debug(s"Creating a new environment reservation")
    validate(environmentReservation)
    val reservation = environmentReservationRepository.create(environmentReservation)
    eventBus.publish(EnvironmentReservationCreatedEvent(reservation))
    reservation
  }

  @Timed
  def updateReservation(environmentReservation: EnvironmentReservation): EnvironmentReservation = {
    logger.debug(s"Updating environment reservation [$environmentReservation]")
    checkArgument(hasText(environmentReservation.getId), "ID is required")
    validate(environmentReservation)
    val reservation = environmentReservationRepository.update(environmentReservation)
    eventBus.publish(EnvironmentReservationUpdatedEvent(reservation))
    reservation
  }

  @Timed
  def deleteReservation(environmentReservationId: String): Unit = {
    logger.debug(s"Deleting environment reservation [$environmentReservationId]")
    val reservation = environmentReservationRepository.findById(environmentReservationId)
    environmentReservationRepository.delete(environmentReservationId)
    eventBus.publish(EnvironmentReservationDeletedEvent(reservation))
  }

  @Timed
  def existsReservation(environmentId: String, applicationId: String): Boolean = {
    logger.debug(s"Checking reservation for environmentId [$environmentId] and applicationId [$applicationId]")
    environmentReservationRepository.existsByEnvironmentIdAndApplicationIdAndDate(environmentId, applicationId, new Date())
  }

  @Timed
  def nearestComingReservation(environmentId: String, applicationId: String): Date = {
    logger.debug(s"Checking nearest coming reservation for environmentId [$environmentId] and applicationId [$applicationId]")
    environmentReservationRepository.findNearestComingByEnvironmentIdAndApplicationIdAndDate(environmentId, applicationId, new Date()).orNull
  }

  private def validate(environmentReservation: EnvironmentReservation): Unit = {
    checkArgument(Option(environmentReservation.getEnvironment).exists(env => hasText(env.getId)), "Environment ID is required")
    validateDates(environmentReservation)
    validateNote(environmentReservation)
    validateApplications(environmentReservation)
  }

  private def validateDates(environmentReservation: EnvironmentReservation): Unit = {
    val startDate = environmentReservation.getStartDate
    val endDate = environmentReservation.getEndDate
    checkArgument(startDate != null, "Start date is required")
    checkArgument(endDate != null, "End date is required")
    checkArgument(startDate.before(endDate), "Start date must be before the end date")
  }

  private def validateNote(environmentReservation: EnvironmentReservation): Unit = {
    if (environmentReservation.getNote != null) {
      checkArgument(hasText(environmentReservation.getNote), "Note cannot be blank")
    }
  }

  private def validateApplications(environmentReservation: EnvironmentReservation): Unit = {
    if (!CollectionUtils.isEmpty(environmentReservation.getApplications)) {
      val allowedAppIds = applicationService
        .getApplicationsDeployableOnEnvironment(environmentReservation.getEnvironment.getId)
        .asScala
        .map(_.getId)
        .toSet
      val givenAppIds = environmentReservation.getApplications.asScala.map(_.getId).toSet
      val diff = givenAppIds diff allowedAppIds
      if (diff.nonEmpty) {
        val (first10, rest) = diff.splitAt(10)
        throw new IncorrectArgumentException(
          s"""
             |The following applications are not deployable on environment [${environmentReservation.getEnvironment.getTitle} - ${environmentReservation.getEnvironment.getId}]:
             |[${first10.mkString(", ")}] ${if (rest.nonEmpty) ", and more"}
             |""".stripMargin
        )
      }
    }
  }
}
