package com.xebialabs.xlrelease.environments.service

import com.xebialabs.xlrelease.domain.{BaseConfiguration, Task, TaskContainer}
import com.xebialabs.xlrelease.repository.Ids.findFolderId
import com.xebialabs.xlrelease.service.TaskCopyHandler
import com.xebialabs.xlrelease.service.TaskService.{TASK_COPY_MESSAGE_DELIMITER, TASK_COPY_MESSAGE_ITEM_FORMAT_1}
import org.springframework.beans.factory.annotation.Autowired
import org.springframework.stereotype.Component

import java.{lang, util}
import scala.jdk.CollectionConverters.CollectionHasAsScala
import scala.util.Try

@Component
class DeploymentFacetTaskCopyHandler @Autowired()(applicationService: ApplicationService,
                                                  environmentService: EnvironmentService) extends TaskCopyHandler {

  override def handle(taskToCopy: Task, targetContainer: TaskContainer, messages: lang.StringBuilder): Unit = {
    val targetFolderId = findFolderId(targetContainer.getId)

    validateFolderReference("applicationId", "Applications", taskToCopy, targetFolderId, messages)(applicationService.findApplicationById)
    validateFolderReference("environmentId", "Environments", taskToCopy, targetFolderId, messages)(environmentService.findEnvironmentById)
  }

  private def validateFolderReference(field: String, section: String, taskToCopy: Task, targetFolderId: String, messages: lang.StringBuilder)
                                     (getEntity: String => BaseConfiguration): Unit = {
    val missingEntities: util.Set[String] = new util.HashSet[String]
    taskToCopy.getFacets.asScala
      .filter(facet => facet.hasProperty(field) && facet.getProperty(field) != null)
      .map(facet => (facet, Try(getEntity(facet.getProperty(field))).toOption))
      .foreach {
        case (facet, Some(entity)) if entity.getFolderId != null && entity.getFolderId != targetFolderId => // currently apps/envs do not have hierarchy...
          facet.setProperty(field, null)
          missingEntities.add(entity.getTitle)
        case _ =>
      }

    if (!missingEntities.isEmpty) {
      initMissingDataSectionMessage(messages)
      messages.append(s"\n$section: \n")
      missingEntities.forEach { env => messages.append(String.format(TASK_COPY_MESSAGE_ITEM_FORMAT_1, env)) }
      messages.append("\n")
    }
  }

  override def getOrder: Int = 0

  private def initMissingDataSectionMessage(messages: lang.StringBuilder): Unit = {
    val missingDataString = "**Missing data in destination.** Create or update the following:\n"
    if (!messages.toString.contains(missingDataString)) messages.append(TASK_COPY_MESSAGE_DELIMITER).append(missingDataString)
  }
}