package com.xebialabs.xlrelease.environments.repository.sql

import com.xebialabs.deployit.exception.NotFoundException
import com.xebialabs.xlrelease.api.v1.filter.{ApplicationEnvironmentFilterScope, ApplicationFilters}
import com.xebialabs.xlrelease.db.sql.SqlBuilder.Dialect
import com.xebialabs.xlrelease.db.sql.transaction.{IsReadOnly, IsTransactional}
import com.xebialabs.xlrelease.domain.environments.Application
import com.xebialabs.xlrelease.environments.repository.sql.persistence.builder.{ApplicationSqlBuilder, ColumnAliases}
import com.xebialabs.xlrelease.environments.repository.sql.persistence.{ApplicationPersistence, EnvironmentPersistence}
import com.xebialabs.xlrelease.environments.repository.{ApplicationRepository, EnvironmentLabelRepository, EnvironmentStageRepository}
import com.xebialabs.xlrelease.repository.Page
import com.xebialabs.xlrelease.repository.sql.SqlRepositoryAdapter
import com.xebialabs.xlrelease.repository.sql.persistence.CiId.CiId
import com.xebialabs.xlrelease.repository.sql.persistence.FolderPersistence
import com.xebialabs.xlrelease.service.CiIdService
import grizzled.slf4j.Logging
import org.springframework.beans.factory.annotation.{Autowired, Qualifier}
import org.springframework.stereotype.Repository

import scala.jdk.CollectionConverters._

@IsTransactional
@Repository
class SqlApplicationRepository @Autowired()(implicit val ciIdService: CiIdService,
                                            @Qualifier("xlrRepositorySqlDialect") implicit val dialect: Dialect,
                                            implicit val repositoryAdapter: SqlRepositoryAdapter,
                                            implicit val environmentPersistence: EnvironmentPersistence,
                                            implicit val environmentLabelRepository: EnvironmentLabelRepository,
                                            implicit val environmentStageRepository: EnvironmentStageRepository,
                                            applicationPersistence: ApplicationPersistence,
                                            folderPersistence: FolderPersistence
                                           )
  extends Logging with ApplicationRepository {

  @IsReadOnly
  def searchApplications(applicationFilters: ApplicationFilters, page: Page): Seq[Application] = {
    val query = ApplicationSqlBuilder()
      .select()
      .orderBy(ColumnAliases.Applications.CI_UID)
      .limitAndOffset(page.resultsPerPage, page.offset)

    Option(applicationFilters).map { f =>
      Option(f.getTitle).foreach(query.withTitleLike)
      query.applyFolderFiltering(Option(f.getScope), Option(f.getFolderId).map(folderPersistence.getUid))
      Option(f.getEnvironments).foreach(envs => query.withEnvironmentTitles(envs.asScala.toSeq))
    }.getOrElse(query.applyFolderFiltering(Some(ApplicationEnvironmentFilterScope.GLOBAL), None))


    if (applicationFilters != null && applicationFilters.getEnvironments != null) {
      query.withEnvironmentTitles(applicationFilters.getEnvironments.asScala.toSeq)
    }

    applicationPersistence.search(query.build()).map(mapApplicationContent)
  }

  @IsReadOnly
  override def findApplicationById(applicationId: String): Application =
    applicationPersistence.findById(applicationId).map(mapApplicationContent)
      .getOrElse(throw new NotFoundException(s"Application [$applicationId] not found"))

  @IsReadOnly
  override def findApplicationByTitle(title: String): Application =
    applicationPersistence.findByTitle(title).map(mapApplicationContent)
      .getOrElse(throw new NotFoundException(s"Application with title [$title] not found"))

  @IsReadOnly
  def findInFolderByCorrelationId(folderId: String, applicationUid: String): Option[Application] =
    applicationPersistence.findInFolderByCorrelationId(folderId, applicationUid).map(mapApplicationContent)

  override def createApplication(application: Application): Application = {
    val appId = applicationPersistence.insert(application)
    findApplicationById(appId)
  }

  override def updateApplication(application: Application): Application = {
    applicationPersistence.update(application)
    findApplicationById(application.getId)
  }

  override def deleteApplication(applicationId: String): Unit = {
    applicationPersistence.delete(applicationId)
  }

  override def getApplicationsDeployableOnEnvironment(environmentId: String): Seq[Application] = {
    val environment = environmentPersistence.findById(environmentId)
      .getOrElse(throw new NotFoundException(s"Environment with ID [$environmentId] not found"))
    val baseQuery = ApplicationSqlBuilder()
      .withEnvironmentIdOrNull(environmentId)
    val query = Option(environment.folderId)
      .fold(baseQuery.withNoFolder())(folderId => baseQuery.withFolder(folderPersistence.getUid(folderId)))
      .build()
    applicationPersistence.search(query).map(mapApplicationContent)
  }

  override def fetchApplications(applicationIds: List[CiId]): Seq[Application] =
    applicationPersistence.fetchApplications(applicationIds).map(mapApplicationContent)

  override def fetchApplicationsByFolderId(folderId: String): Seq[Application] =
    applicationPersistence.fetchApplicationsByFolderId(folderId).map(mapApplicationContent)

  override def findByEnvironment(environmentId: String): List[String] = {
    applicationPersistence.findAppsWithEnv(environmentId)
  }

}
