import React from 'react';
import { ReactWrapper } from 'enzyme';
import { ApplicationListItemComponent } from './application-list-item-component';
import { mountWithStoreAndTheme } from '../../../../../../../../../../core/xlr-ui/tests/unit/testing-utils';
import { application } from '../__mocks__/application-list.mocks';
import { applicationLists } from '../ducks/application-list.reducer';
import { Provider } from 'react-redux';
import { Application } from '../../../application-list.types';

const { editApplication } = applicationLists.actions;

describe('Application list item component', () => {
    let wrapper: ReactWrapper;
    const dispatch = jest.fn();
    const onDelete = jest.fn();

    const mount = (app: Application = application) => {
        wrapper = mountWithStoreAndTheme(<ApplicationListItemComponent {...{ application: app, onDelete }} />, dispatch);
    };

    beforeEach(() => {
        mount();
    });

    afterEach(() => {
        wrapper.unmount();
    });

    it('should render properly', () => {
        expect(wrapper.find(Provider)).toExist();
        expect(wrapper.find(ApplicationListItemComponent)).toExist();
    });

    it('should create default state', () => {
        const panel = wrapper.find('.application-panel');
        expect(panel.exists()).toBeTruthy();
    });

    it('should handle properly callbacks', () => {
        const panel = wrapper.find('.application-panel');
        panel.simulate('click');
        expect(dispatch).toBeCalledWith(editApplication({ appId: application.id, folderId: application.folderId }));

        const editLink = wrapper.find('[data-test="edit-action"]');
        expect(editLink.text()).toBe('Edit');
        editLink.simulate('click');
        expect(dispatch).toBeCalledWith(editApplication({ appId: application.id, folderId: application.folderId }));
        const deleteLink = wrapper.find('[data-test="delete-action"]');
        deleteLink.simulate('click');
        expect(onDelete).toHaveBeenCalledTimes(1);
        expect(deleteLink.text()).toBe('Delete');
    });

    it('should render application with environments', () => {
        const application1 = {
            id: 'Application1',
            title: 'App1',
            environments: [
                {
                    stage: { id: 'EnvironmentStage1', title: 'Test' },
                    id: 'Environment1',
                    title: 'dev01-pwd.com',
                    description: 'development environment',
                    folderId: 'Folder42',
                    labels: [],
                },
                {
                    stage: { id: 'EnvironmentStage1', title: 'Test' },
                    id: 'Environment2',
                    title: 'dev02-pwd.com',
                    description: 'development environment',
                    folderId: 'Folder42',
                    labels: [],
                },
            ],
            folderId: 'Folder42',
        };
        mount(application1);
        const environmentsLinked = wrapper.find('.application-item-environments > span');
        expect(environmentsLinked.text()).toBe('dev01-pwd.com, dev02-pwd.com');
    });

    it('should render application without environments', () => {
        const environmentsLinked = wrapper.find('.application-item-environments');
        expect(environmentsLinked.text()).toBe('All environments');
    });
});
