package com.xebialabs.xlrelease.environments.repository.sql.persistence.builder

import com.xebialabs.xlrelease.db.sql.SqlBuilder.Dialect
import com.xebialabs.xlrelease.db.sql.{Sql, SqlBuilder, SqlWithParameters}
import com.xebialabs.xlrelease.environments.repository.sql.persistence.toPersistedId
import com.xebialabs.xlrelease.repository.sql.persistence.CiId.CiId

import java.util.Date

object EnvironmentsWithReservationsSqlBuilder {
  def apply(aliasPrefix: String = "")(implicit dialect: Dialect): EnvironmentsWithReservationsSqlBuilder = {
    new EnvironmentsWithReservationsSqlBuilder(aliasPrefix)
  }
}

class EnvironmentsWithReservationsSqlBuilder(aliasPrefix: String)(implicit dialect: Dialect) extends SqlBuilder[EnvironmentsWithReservationsSqlBuilder] {
  private val resAlias = makeAlias(aliasPrefix, "res")
  private val envAlias = makeAlias(aliasPrefix, "env")

  private lazy val resBuilder = ReservationSqlBuilder(resAlias)
  private lazy val envBuilder = EnvironmentSqlBuilder(envAlias)

  // Ignore environment lists inside applications in this query
  override def build(): SqlWithParameters = {
    val (environmentQuery, envParams) = envBuilder.select().build()
    val (reservationQuery, baseResParams) = resBuilder.select().build()
    val stmt =
      s"""
         |SELECT
         | $envAlias.${ColumnAliases.Environments.ID} ${ColumnAliases.Environments.ID}
         | ,$envAlias.${ColumnAliases.Environments.FOLDER_ID} ${ColumnAliases.Environments.FOLDER_ID}
         | ,$envAlias.${ColumnAliases.Environments.FOLDER_PATH} ${ColumnAliases.Environments.FOLDER_PATH}
         | ,$envAlias.${ColumnAliases.Environments.TITLE} ${ColumnAliases.Environments.TITLE}
         | ,$envAlias.${ColumnAliases.Environments.DESCRIPTION} ${ColumnAliases.Environments.DESCRIPTION}
         | ,$envAlias.${ColumnAliases.Environments.CONTENT} ${ColumnAliases.Environments.CONTENT}
         | ,$envAlias.${ColumnAliases.EnvStages.ID} ${ColumnAliases.EnvStages.ID}
         | ,$envAlias.${ColumnAliases.EnvStages.TITLE} ${ColumnAliases.EnvStages.TITLE}
         | ,$envAlias.${ColumnAliases.EnvLabels.ID} ${ColumnAliases.EnvLabels.ID}
         | ,$envAlias.${ColumnAliases.EnvLabels.TITLE} ${ColumnAliases.EnvLabels.TITLE}
         | ,$envAlias.${ColumnAliases.EnvLabels.COLOR} ${ColumnAliases.EnvLabels.COLOR}
         | ,$resAlias.${ColumnAliases.EnvReservations.ID} ${ColumnAliases.EnvReservations.ID}
         | ,$resAlias.${ColumnAliases.EnvReservations.START_DATE} ${ColumnAliases.EnvReservations.START_DATE}
         | ,$resAlias.${ColumnAliases.EnvReservations.END_DATE} ${ColumnAliases.EnvReservations.END_DATE}
         | ,$resAlias.${ColumnAliases.EnvReservations.NOTE} ${ColumnAliases.EnvReservations.NOTE}
         | ,$resAlias.${ColumnAliases.Applications.ID} ${ColumnAliases.Applications.ID}
         | ,$resAlias.${ColumnAliases.Applications.FOLDER_ID} ${ColumnAliases.Applications.FOLDER_ID}
         | ,$resAlias.${ColumnAliases.Applications.FOLDER_PATH} ${ColumnAliases.Applications.FOLDER_PATH}
         | ,$resAlias.${ColumnAliases.Applications.TITLE} ${ColumnAliases.Applications.TITLE}
         | ,$resAlias.${ColumnAliases.Applications.CONTENT} ${ColumnAliases.Applications.CONTENT}
         | FROM (${indent(environmentQuery)}) $envAlias
         | LEFT JOIN (${indent(reservationQuery)}) $resAlias
         |  ON $envAlias.${ColumnAliases.Environments.CI_UID} = $resAlias.${ColumnAliases.EnvReservations.ENVIRONMENT_UID}
      """.stripMargin
    // Merge into 1 big sql with parameters
    super.select(stmt)
    super.orderBy(s"$resAlias.${ColumnAliases.EnvReservations.START_DATE} ASC")
    val fullSql = super.build()

    fullSql._1 -> (envParams ++ baseResParams ++ fullSql._2)
  }

  def select(): EnvironmentsWithReservationsSqlBuilder = this

  def withApplicationTitles(titles: Iterable[String]): EnvironmentsWithReservationsSqlBuilder = {
    resBuilder.withApplicationTitles(titles)
    this
  }

  def from(date: Date): EnvironmentsWithReservationsSqlBuilder = {
    resBuilder.from(date)
    this
  }

  def to(date: Date): EnvironmentsWithReservationsSqlBuilder = {
    resBuilder.to(date)
    this
  }

  // todo specialized query for just reservations
  def withReservationId(id: CiId): EnvironmentsWithReservationsSqlBuilder = {
    conditions += Sql(s"$resAlias.${ColumnAliases.EnvReservations.ID} = ?", Seq(id))
    this
  }

  def withEnvironmentTitleLike(title: String): EnvironmentsWithReservationsSqlBuilder = {
    envBuilder.withTitleLike(title)
    this
  }

  def withStageTitles(titles: Iterable[String]): EnvironmentsWithReservationsSqlBuilder = {
    envBuilder.withStageTitles(titles)
    this
  }

  def withLabelTitles(titles: Iterable[String]): EnvironmentsWithReservationsSqlBuilder = {
    envBuilder.withLabelTitles(titles)
    this
  }

  def withEnvironmentId(environmentId: CiId): EnvironmentsWithReservationsSqlBuilder = {
    envBuilder.withEnvironmentId(toPersistedId(environmentId))
    this
  }

  // Pagination by environments
  override def limitAndOffset(limit: Long, offset: Long): EnvironmentsWithReservationsSqlBuilder = {
    envBuilder.limitAndOffset(limit, offset)
    this
  }

  override def orderBy(column: String): EnvironmentsWithReservationsSqlBuilder = {
    envBuilder.orderBy(column)
    this
  }

  override def newInstance: EnvironmentsWithReservationsSqlBuilder = new EnvironmentsWithReservationsSqlBuilder(aliasPrefix)

}
