import React, { useState } from 'react';
import isEmpty from 'lodash/isEmpty';
import WidgetLinkedEnvironments from './../../../shared/widgets/widget-linked-environments';
import { itemAlreadyDefined } from '../../../../../../../../../../../core/xlr-ui/app/react/utils/validators';
import './../../../../styles/common.less';
import './application-form-component.less';
import { Application, Environment } from '../../../../application-list.types';
import { DotInputText, DotThemeProvider } from '@digital-ai/dot-components';

export interface ApplicationFormProps {
    applicationList: Array<Application>;
    applicationModel: Application;
    onFormChange: (app: Application) => void;
    onValidationChange: (valid: boolean) => void;
}

const nameFormMetadata = {
    label: 'Name',
    placeholder: 'Add name...',
    description: 'Name of the application',
    required: true,
};

const findApplicationById = (applications: Array<Application>, id: string) => {
    return applications.find((e) => e.id === id) || {};
};
export const ApplicationFormComponent = ({ applicationList, applicationModel, onFormChange, onValidationChange }: ApplicationFormProps) => {
    const [isItemAlreadyDefined, setItemAlreadyDefined] = useState(false);
    const errorMessage = 'Application already exists.';

    const setFormValidation = (valid: boolean) => {
        onValidationChange(valid);
    };
    const onNameChange = (title: string) => {
        if (title !== applicationModel.title) {
            onFormChange({
                ...applicationModel,
                title,
            });
        }
        const error = itemAlreadyDefined(applicationList, applicationModel.id, 'title', title);
        setItemAlreadyDefined(error);
        setFormValidation(!error);
    };

    const onEnvironmentDeleted = (environment: Environment) => {
        const environments = applicationModel.environments.filter((e) => e.id !== environment.id);
        onFormChange({
            ...applicationModel,
            environments,
        });
    };

    const app = findApplicationById(applicationList, applicationModel.id) as Application;
    const subtitle = isEmpty(app) ? 'New Application' : app.title;

    return (
        <div className="application-edit-form">
            <h4 className="page-title">Applications / {subtitle}</h4>
            <div className="application-form">
                <DotThemeProvider>
                    <DotInputText
                        autoFocus={true}
                        className={'form-input'}
                        defaultValue={''}
                        error={isItemAlreadyDefined}
                        helperText={isItemAlreadyDefined ? errorMessage : ''}
                        id={'app-form-name'}
                        label={nameFormMetadata.label}
                        maxLength={255}
                        name={'app-form-name'}
                        onChange={(e) => onNameChange(e.target.value)}
                        persistentLabel={true}
                        placeholder={nameFormMetadata.placeholder}
                        required={nameFormMetadata.required}
                        value={applicationModel.title}
                    />
                    <WidgetLinkedEnvironments environments={applicationModel.environments} onDelete={onEnvironmentDeleted} />
                </DotThemeProvider>
            </div>
        </div>
    );
};
