import React from 'react';

import { initEnvLabel, LabelListComponent } from './label-list-component';
import { LabelListHeaderComponent } from './label-list-header-component';
import LabelItem from '../../shared/label/label-item';
import { LabelModalFormComponent } from './label-modal-form-component';
import { NoData } from '../../../../../../../../../../core/xlr-ui/app/react/components';
import { mountWithStoreAndTheme, preparePortalContainer, ReactWrapper } from '../../../../../../../../../../core/xlr-ui/tests/unit/testing-utils';
import { labelsState } from '../ducks/labels.reducer';
import { RootState } from '../../../../../../../../../../core/xlr-ui/app/js/store.types';
import { labels } from '../__mocks__/environment-list.mocks';
import { DotConfirmationDialog } from '@digital-ai/dot-components';

const { searchLabels, saveLabel, deleteLabel } = labelsState.actions;

describe('Label list component', () => {
    let wrapper: ReactWrapper;
    const dispatch = jest.fn();
    const stateWithLabels: RootState = {
        labelsState: {
            labelList: labels,
        },
    };
    const stateWithNoLabels: RootState = {
        labelsState: {
            labelList: [],
        },
    };

    const mount = (state: RootState) => {
        wrapper = mountWithStoreAndTheme(<LabelListComponent />, dispatch, state);
    };

    beforeEach(() => {
        preparePortalContainer();
    });

    it('should render with model', () => {
        mount(stateWithLabels);
        const header = wrapper.find(LabelListHeaderComponent);
        expect(header.exists()).toBeTruthy();
        expect(wrapper.find('h4').text()).toBe('Labels (2)');
        expect(wrapper.find(NoData).length).toBe(0);

        const items = wrapper.find(LabelItem);
        expect(items).toHaveLength(2);

        const modal = wrapper.find(LabelModalFormComponent);
        expect(modal.exists()).toBeTruthy();
    });

    it('should render NoData placeholder for empty list', () => {
        mount(stateWithNoLabels);
        expect(wrapper.find('h4').text()).toBe('Labels (0)');
        expect(wrapper.find(NoData).length).toBe(1);
    });

    it('should handle callbacks', () => {
        mount(stateWithLabels);

        const header = wrapper.find(LabelListHeaderComponent);
        header.invoke('onCreate')();
        header.invoke('onFilterChange')('new title');
        expect(dispatch).toHaveBeenCalledWith(searchLabels('new title'));

        const modal = wrapper.find(LabelModalFormComponent);
        expect(modal.prop('modalIsOpen')).toBeTruthy();

        const lbl = { id: '', color: 'blue', title: 'new' };
        modal.invoke('onConfirmModal')(lbl, true);
        expect(dispatch).toHaveBeenCalledWith(saveLabel(lbl));

        modal.invoke('onOpenDelete')();
        expect(wrapper.find(DotConfirmationDialog)).toHaveLength(1);
    });

    it('should delete label', () => {
        mount(stateWithLabels);
        expect(dispatch).toBeCalledWith(searchLabels(''));
        const modal = wrapper.find(LabelModalFormComponent);
        modal.invoke('onOpenDelete')();
        const dialog = wrapper.find(DotConfirmationDialog);
        expect(dialog).toHaveLength(1);
        dialog.invoke('onSubmit')();
        expect(dispatch).toBeCalledWith(deleteLabel(initEnvLabel));
    });
});
