import React, { useEffect, useState } from 'react';

import { ApplicationListHeaderComponent } from './application-list-header-component';
import { ApplicationListItemComponent } from './application-list-item-component';
import { AlertMessage, alertType, HelpBlock, NoData } from '../../../../../../../../../../core/xlr-ui/app/react/components';

import './../../../styles/common.less';
import './application-list-component.less';
import { Application } from '../../../application-list.types';
import { useAppDispatch, useAppSelector } from '../../../../../../../../../../core/xlr-ui/app/js/hooks';
import { applicationLists } from '../ducks/application-list.reducer';
import { getApplicationForDelete, getApplicationsSelector } from '../ducks/application-list.selectors';
import { DotConfirmationDialog, DotIcon, DotThemeProvider, DotTypography } from '@digital-ai/dot-components';

export interface ApplicationListProps {
    folderId: string;
}

const { searchApplications, setApplicationToDelete, deleteApplication } = applicationLists.actions;

export const ApplicationListComponent = ({ folderId }: ApplicationListProps) => {
    const dispatch = useAppDispatch();

    useEffect(() => {
        dispatch(searchApplications({ folderId, title: '' }));
    }, []);

    const [deleteModalIsOpen, setDeleteModalIsOpen] = useState(false);
    const applications = useAppSelector(getApplicationsSelector);
    const applicationToDelete = useAppSelector(getApplicationForDelete);

    const helpMessage = `Use Applications to logically group applications and the environments they must be deployed to.`;

    const showDeleteModal = (application: Application) => {
        setDeleteModalIsOpen(true);
        dispatch(setApplicationToDelete(application));
    };

    const onCloseDeleteModal = (action: string) => {
        if (action === 'delete' && applicationToDelete !== undefined) {
            dispatch(deleteApplication(applicationToDelete));
        }
        setDeleteModalIsOpen(false);
    };

    const renderDeleteModal = () => {
        return (
            <DotConfirmationDialog
                className="xl-default-modal"
                message={
                    <div>
                        <AlertMessage alertType={alertType.WARNING}>
                            <p>Deleting an application can lead to missing data for connected resources on reporting and environment scheduling.</p>
                        </AlertMessage>
                        <p>
                            You are about to delete application
                            <strong>{applicationToDelete ? ' ' + applicationToDelete.title : 'No application selected'}</strong>.
                        </p>
                    </div>
                }
                onCancel={() => onCloseDeleteModal('cancel')}
                onSubmit={() => onCloseDeleteModal('delete')}
                open={deleteModalIsOpen}
                submitButtonProps={{ className: 'deleteBtn', label: 'Delete Application', startIcon: <DotIcon iconId="delete" />, type: 'destructive' }}
                title={<DotTypography variant="h1">Delete application</DotTypography>}
            />
        );
    };

    const renderHeader = () => (
        <div className="application-list-header">
            <span className="col-xs-5">Name</span>
            <span className="col-xs-5">Environment</span>
            <div className="col-xs-2">
                <span className="pull-right action-section">Actions</span>
            </div>
        </div>
    );

    const renderList = () => {
        return (
            <React.Fragment>
                {renderDeleteModal()}
                {renderHeader()}
                {applications.map((app) => (
                    <ApplicationListItemComponent application={app} key={app.id} onDelete={showDeleteModal} />
                ))}
            </React.Fragment>
        );
    };

    return (
        <DotThemeProvider>
            <ApplicationListHeaderComponent folderId={folderId} />
            <div className="header-shadow"></div>
            <div className="application-list-component">
                <h4 className="page-title">Applications ({applications.length})</h4>
                <HelpBlock className="help-message-container">{helpMessage}</HelpBlock>
                {applications.length ? renderList() : <NoData message="No applications found." />}
            </div>
        </DotThemeProvider>
    );
};
