import React, { useEffect, useState } from 'react';

import { LabelListHeaderComponent } from './label-list-header-component';
import LabelItem from './../../shared/label/label-item';
import { LabelModalFormComponent } from './label-modal-form-component';
import { AlertMessage, HelpBlock, NoData, alertType } from '../../../../../../../../../../core/xlr-ui/app/react/components';

import './../../../styles/common.less';
import { Label } from '../../../application-list.types';
import { labelsState } from '../ducks/labels.reducer';
import { useAppDispatch, useAppSelector } from '../../../../../../../../../../core/xlr-ui/app/js/hooks';
import { getLabelsList } from '../ducks/environment-list.selectors';
import { DotConfirmationDialog, DotIcon, DotThemeProvider, DotTypography } from '@digital-ai/dot-components';

const { searchLabels, saveLabel, deleteLabel } = labelsState.actions;

const helpMessage = `Use Labels to identify your environments. You also use Labels to filter the environment scheduler and dashboards.`;
export const initEnvLabel = {
    id: null,
    color: '',
    title: '',
} as Label;
export const LabelListComponent = () => {
    const dispatch = useAppDispatch();

    const [isModalVisible, setModalVisible] = useState(false);
    const [labelModel, setLabelModel] = useState(initEnvLabel);
    const [isDeleteModal, setDeleteModal] = useState(false);

    useEffect(() => {
        dispatch(searchLabels(''));
    }, []);

    const showCreateModal = () => {
        setModalVisible(true);
        setLabelModel(initEnvLabel);
    };

    const showEditModal = (label: Label) => {
        setModalVisible(true);
        setLabelModel(label);
    };
    const onFilterChange = (title: string) => {
        dispatch(searchLabels(title));
    };

    const onConfirmModal = (envLbl: Label, isCreateAnother: boolean) => {
        dispatch(saveLabel(envLbl));
        onCancelModal();

        const envLabelState = initEnvLabel;
        setLabelModel(envLabelState);
        if (isCreateAnother) {
            envLabelState.color = envLbl.color;
            showEditModal(envLabelState);
        }
    };

    const onShowDeleteDialog = () => {
        setModalVisible(false);
        setDeleteModal(true);
    };

    const onCancelModal = () => {
        setModalVisible(false);
        setLabelModel(initEnvLabel);
    };

    const onConfirmDeleteModal = (envLbl: Label) => {
        dispatch(deleteLabel(envLbl));
        onCancelModal();
    };

    const submitDelete = () => {
        setDeleteModal(false);
        onConfirmDeleteModal(labelModel);
    };

    const cancelDelete = () => {
        setDeleteModal(false);
        setModalVisible(true);
    };

    const renderDelete = () => {
        return (
            <DotConfirmationDialog
                className="xl-default-modal"
                message={
                    <div>
                        <AlertMessage alertType={alertType.WARNING}>
                            <p>Deleting a label can lead to missing data for connected resources on reporting and environment scheduling.</p>
                        </AlertMessage>
                        <p>
                            You are about to delete label <strong>{labelModel.title}</strong>.
                        </p>
                    </div>
                }
                onCancel={cancelDelete}
                onSubmit={submitDelete}
                open={isDeleteModal}
                submitButtonProps={{ className: 'deleteBtn', label: 'Delete Label', startIcon: <DotIcon iconId="delete" />, type: 'destructive' }}
                title={<DotTypography variant="h1">Delete label</DotTypography>}
            />
        );
    };

    const renderEnvLabel = (envLbl: Label) => {
        return <LabelItem envLabel={envLbl} isEditable={true} key={envLbl.id} onClickLabel={showEditModal} />;
    };

    const labelsList = useAppSelector(getLabelsList);
    const renderLabels = () => {
        const labels = labelsList.map(renderEnvLabel);

        return (
            <div className="margin">
                <h4 className="page-title">Labels ({labels.length})</h4>
                <HelpBlock className="help-message-container">{helpMessage}</HelpBlock>
                {labels.length ? labels : <NoData className="no-labels-message" message="No labels found." />}
            </div>
        );
    };

    return (
        <DotThemeProvider>
            <LabelListHeaderComponent onCreate={showCreateModal} onFilterChange={onFilterChange} />
            <div className="header-shadow"></div>
            {renderLabels()}

            {renderDelete()}
            <LabelModalFormComponent
                labelModel={labelModel}
                modalIsOpen={isModalVisible}
                onCancelModal={onCancelModal}
                onConfirmModal={onConfirmModal}
                onOpenDelete={onShowDeleteDialog}
            />
        </DotThemeProvider>
    );
};
