import { call, put, all, takeLatest, throttle, select } from 'redux-saga/effects';
import { SagaIterator } from 'redux-saga';
import applicationListsSaga, {
    goToApplicationEdit,
    goToApplicationCreate,
    deleteApplicationAction,
    scheduleInitHeader,
    searchApplicationsAction,
    toastr,
} from './application-list.saga';
import { applicationLists } from './application-list.reducer';
import { actionTypes as schedulerActionTypes } from '../../../actions/scheduling-actions';
import { httpDELETE, httpPOST } from '../../../../../../../../../../core/xlr-ui/app/features/common/services/http';
import { allApplications, application, filteredApplications } from '../__mocks__/application-list.mocks';
import { getApplicationsSelector } from './application-list.selectors';
import { applicationForm } from './application-form.reducer';
import getAngularService from '../../../../../../../../../../core/xlr-ui/app/features/common/services/angular-accessor';

const { editApplication, createApplication, setApplications, searchApplications, deleteApplication } = applicationLists.actions;

const { onFormClean } = applicationForm.actions;

describe('application-list.saga', () => {
    it('should yield all effects', () => {
        const gen: SagaIterator = applicationListsSaga();
        expect(gen.next().value).toStrictEqual(
            all([
                takeLatest(createApplication, goToApplicationCreate),
                takeLatest(editApplication, goToApplicationEdit),
                takeLatest(deleteApplication, deleteApplicationAction),
                throttle(200, searchApplications, searchApplicationsAction),
                throttle(200, schedulerActionTypes.SCHEDULE_INIT_HEADER_SUCCESS, scheduleInitHeader),
            ]),
        );
        expect(gen.next().done).toBe(true);
    });

    describe('searchApplicationsActions', () => {
        it('should search and set applications', () => {
            const applicationName = '';
            const folderId = '';
            const gen: SagaIterator = searchApplicationsAction({
                payload: { folderId, title: applicationName },
            });

            expect(gen.next({ folderId, applicationName }).value).toMatchObject(
                call(httpPOST, 'api/v1/applications/search', { folderId, title: applicationName }),
            );
            expect(gen.next({ data: allApplications }).value).toStrictEqual(put(setApplications(allApplications)));
            expect(gen.next().done).toBe(true);
        });
    });

    describe('deleteApplicationAction', () => {
        it('should delete application', () => {
            const gen: SagaIterator = deleteApplicationAction({
                payload: application,
                type: 'whatever',
            });
            expect(gen.next().value).toMatchObject(call(httpDELETE, `api/v1/applications/${application.id}`));
            expect(gen.next().value).toStrictEqual(select(getApplicationsSelector));
            expect(gen.next(filteredApplications).value).toStrictEqual(put(setApplications(filteredApplications)));
            expect(gen.next().value).toMatchObject(call(toastr.success, 'Deleted application [Application1]'));
            expect(gen.next().done).toBe(true);
        });
    });

    describe('goToApplicationCreate', () => {
        it('should clean form and change state to applicationCreate without applicationId', () => {
            const $state = { go: jest.fn() };
            const gen: SagaIterator = goToApplicationCreate({
                payload: '',
                type: 'whatever',
            });
            expect(gen.next().value).toStrictEqual(put(onFormClean()));
            expect(gen.next().value).toStrictEqual(call(getAngularService, '$state'));
            expect(gen.next($state).value).toStrictEqual($state.go('applicationCreate'));
            expect(gen.next().done).toBe(true);
        });

        it('should clean form and change state to applicationCreate without applicationId', () => {
            const $state = { go: jest.fn() };
            const gen: SagaIterator = goToApplicationCreate({
                payload: 'nekiid',
                type: 'whatever',
            });
            expect(gen.next().value).toStrictEqual(put(onFormClean()));
            expect(gen.next().value).toStrictEqual(call(getAngularService, '$state'));
            expect(gen.next($state).value).toStrictEqual($state.go('folders.detail.applicationCreate'));
            expect(gen.next().done).toBe(true);
        });
    });

    describe('goToApplicationEdit', () => {
        it('should clean form and change state to applicationCreate with applicationId', () => {
            const $state = { go: jest.fn() };
            const gen: SagaIterator = goToApplicationEdit({
                payload: { folderId: '', appId: 'someId' },
                type: 'string',
            });
            expect(gen.next().value).toStrictEqual(call(getAngularService, '$state'));
            expect(gen.next($state).value).toStrictEqual($state.go('applicationCreate', { applicationId: 'someId' }));
            expect(gen.next().done).toBe(true);
        });

        it('should clean form and change state to applicationCreate inside folder', () => {
            const $state = { go: jest.fn() };
            const gen: SagaIterator = goToApplicationEdit({
                payload: { folderId: 'Folder42', appId: 'someId' },
                type: 'string',
            });
            expect(gen.next().value).toStrictEqual(call(getAngularService, '$state'));
            expect(gen.next($state).value).toStrictEqual($state.go('folders.detail.applicationCreate', { applicationId: 'someId' }));
            expect(gen.next().done).toBe(true);
        });
    });

    describe('schedule init header', () => {
        const gen: SagaIterator = scheduleInitHeader({
            payload: {
                applications: [application],
                labels: [],
                stages: [],
            },
            type: 'whatever',
        });
        expect(gen.next().value).toStrictEqual(put(setApplications([application])));
        expect(gen.next().done).toBeTruthy();
    });
});
