import { all, call, put, takeLatest } from 'redux-saga/effects';
import isEmpty from 'lodash/isEmpty';
import ToastrFactory from '../../../../../../../../../../core/xlr-ui/app/js/util/toastrFactory';
import { PayloadAction } from '@reduxjs/toolkit';
import getAngularService from '../../../../../../../../../../core/xlr-ui/app/features/common/services/angular-accessor';
import { addConfirmLeavePopup, removeConfirmLeavePopup } from '../../../../../../../../../../core/xlr-ui/app/react/utils/sagas';
import { applicationForm } from './application-form.reducer';
import { Application, ApplicationEdit } from '../../../application-list.types';
import { httpGET, httpPOST, httpPUT } from '../../../../../../../../../../core/xlr-ui/app/features/common/services/http';
import { IStateService } from 'angular-ui-router';

export const toastr = ToastrFactory();
const {
    setApplication,
    setApplicationFolder,
    initForm,
    setStages,
    setApplications,
    setEnvironments,
    onFormClean,
    cancelForm,
    onSave,
    onUpdate,
    pristineFormChange,
} = applicationForm.actions;

export function* initApplicationForm(action: PayloadAction<ApplicationEdit>) {
    try {
        const { appId, folderId } = action.payload;
        if (!isEmpty(appId)) {
            const { data: app } = yield call(httpGET, `api/v1/applications/${appId}`);
            yield put(setApplication(app));
            yield put(setApplicationFolder(folderId ? folderId : ''));
        }
        const { data: environments } = yield call(httpPOST, 'api/v1/environments/search', { title: '', folderId });
        yield put(setEnvironments(environments));

        const { data: stages } = yield call(httpPOST, 'api/v1/environments/stages/search', { title: '' });
        yield put(setStages(stages));

        const { data: applications } = yield call(httpPOST, 'api/v1/applications/search', { folderId, title: '' });
        yield put(setApplications(applications));
    } catch (e) {
        // eslint-disable-next-line no-console,angular/log
        yield call(console.error, 'Application form saga: ', e);
    }
}

export function* cancelApplicationForm(action: PayloadAction<string>) {
    const folderId = action.payload;
    const $state: IStateService = yield call(getAngularService, '$state');
    yield call(removeConfirmLeavePopup);
    if (isEmpty(folderId)) {
        $state.go('applicationList');
    } else {
        $state.go('folders.detail.folderApplication');
    }
    yield put(onFormClean);
}

export function* updateApplicationModel(action: PayloadAction<{ app: Application; folderId: string }>) {
    yield put(setApplication(action.payload.app));
    yield put(setApplicationFolder(action.payload.folderId));
}

export function* saveApplication(action: PayloadAction<Application>) {
    try {
        const application = action.payload;
        const environmentIds = application.environments.map((env) => env.id);
        const appForSave = { ...application, environmentIds };
        const $state: IStateService = yield call(getAngularService, '$state');
        yield call(removeConfirmLeavePopup);
        if (isEmpty(application.id)) {
            yield call(httpPOST, 'api/v1/applications', appForSave);
            yield call(toastr.success, `Created application [${action.payload.title}]`);
        } else {
            yield call(httpPUT, `api/v1/applications/${application.id}`, appForSave);
            yield call(toastr.success, `Updated application [${action.payload.title}]`);
        }
        if (isEmpty(application.folderId)) {
            $state.go('applicationList');
        } else {
            $state.go('folders.detail.folderApplication');
        }
    } catch (e) {
        // eslint-disable-next-line no-console,angular/log
        yield call(console.error, 'Application form saga: ', e);
    }
}

export default function* applicationFormSaga() {
    yield all([
        takeLatest(onSave, saveApplication),
        takeLatest(initForm, initApplicationForm),
        takeLatest(onUpdate, updateApplicationModel),
        takeLatest(cancelForm, cancelApplicationForm),
        takeLatest(pristineFormChange, addConfirmLeavePopup),
    ]);
}
