import getAngularService from '../../../../../../../../../../core/xlr-ui/app/features/common/services/angular-accessor';
import { all, call, put, takeLatest } from 'redux-saga/effects';
import { addConfirmLeavePopup, removeConfirmLeavePopup } from '../../../../../../../../../../core/xlr-ui/app/react/utils/sagas';
import { applicationForm } from './application-form.reducer';
import { allApplications, environment, environments, stage } from '../__mocks__/application-list.mocks';
import { SagaIterator } from 'redux-saga';
import applicationFormSaga, { initApplicationForm, saveApplication, updateApplicationModel, cancelApplicationForm, toastr } from './application-form.saga';
import { httpGET, httpPOST, httpPUT } from '../../../../../../../../../../core/xlr-ui/app/features/common/services/http';

const {
    initForm,
    onSave,
    onUpdate,
    pristineFormChange,
    cancelForm,
    onFormClean,
    setApplication,
    setApplications,
    setApplicationFolder,
    setEnvironments,
    setStages,
} = applicationForm.actions;

describe('application-form.saga', () => {
    it('should yield all effects', () => {
        const gen: SagaIterator = applicationFormSaga();
        expect(gen.next().value).toStrictEqual(
            all([
                takeLatest(onSave, saveApplication),
                takeLatest(initForm, initApplicationForm),
                takeLatest(onUpdate, updateApplicationModel),
                takeLatest(cancelForm, cancelApplicationForm),
                takeLatest(pristineFormChange, addConfirmLeavePopup),
            ]),
        );
        expect(gen.next().done).toBe(true);
    });

    describe('updateApplicationModel', () => {
        it('should update application and set application folder', () => {
            const application = { id: 'Application1', title: 'App1', folderId: '', environments: [environment] };
            const gen: SagaIterator = updateApplicationModel({
                payload: {
                    folderId: '',
                    app: application,
                },
                type: 'whatever',
            });
            expect(gen.next({ application }).value).toStrictEqual(put(setApplication(application)));
            expect(gen.next({ folderId: '' }).value).toStrictEqual(put(setApplicationFolder('')));
            expect(gen.next().done).toBe(true);
        });
    });

    describe('saveApplication', () => {
        it('should save updated application and go to list page', () => {
            const $state = { go: jest.fn() };
            const application = { id: 'Application1', title: 'App1', folderId: 'Folder42', environments: [environment] };
            const applicationToSave = {
                id: 'Application1',
                title: 'App1',
                folderId: 'Folder42',
                environments: [environment],
                environmentIds: ['Environment1'],
            };
            const gen: SagaIterator = saveApplication({
                payload: application,
                type: 'whatever',
            });
            expect(gen.next().value).toStrictEqual(call(getAngularService, '$state'));
            expect(gen.next($state).value).toStrictEqual(call(removeConfirmLeavePopup));
            expect(gen.next(applicationToSave).value).toStrictEqual(call(httpPUT, `api/v1/applications/${application.id}`, applicationToSave));
            expect(gen.next(application).value).toStrictEqual(call(toastr.success, 'Updated application [App1]'));
            expect(gen.next(application).value).toStrictEqual($state.go('folders.detail.folderApplication'));
            expect(gen.next().done).toBe(true);
        });

        it('should create application and go to list page', () => {
            const $state = { go: jest.fn() };
            const application = { id: '', title: 'App1', folderId: '', environments: [environment] };
            const applicationToSave = { id: '', title: 'App1', folderId: '', environments: [environment], environmentIds: ['Environment1'] };
            const gen: SagaIterator = saveApplication({
                payload: application,
                type: 'whatever',
            });
            expect(gen.next().value).toStrictEqual(call(getAngularService, '$state'));
            expect(gen.next($state).value).toStrictEqual(call(removeConfirmLeavePopup));
            expect(gen.next(applicationToSave).value).toStrictEqual(call(httpPOST, 'api/v1/applications', applicationToSave));
            expect(gen.next(application).value).toStrictEqual(call(toastr.success, 'Created application [App1]'));
            expect(gen.next(application).value).toStrictEqual($state.go('applicationList'));
            expect(gen.next().done).toBe(true);
        });

        it('should create application and go to list page on folder', () => {
            const $state = { go: jest.fn() };
            const application = { id: '', title: 'App1', environments: [environment], folderId: 'Folder42' };
            const applicationToSave = { id: '', title: 'App1', environments: [environment], environmentIds: ['Environment1'], folderId: 'Folder42' };
            const gen: SagaIterator = saveApplication({
                payload: application,
                type: 'whatever',
            });
            expect(gen.next().value).toStrictEqual(call(getAngularService, '$state'));
            expect(gen.next($state).value).toStrictEqual(call(removeConfirmLeavePopup));
            expect(gen.next().value).toStrictEqual(call(httpPOST, 'api/v1/applications', applicationToSave));
            expect(gen.next().value).toStrictEqual(call(toastr.success, 'Created application [App1]'));
            expect(gen.next().value).toStrictEqual($state.go('folders.detail.folderApplication'));
            expect(gen.next().done).toBe(true);
        });
    });

    describe('initApplicationForm', () => {
        it('should load all the data needed on init while editing application', () => {
            const applicationId = 'Application1';
            const folderId = '';
            const application = { id: applicationId, title: 'App1', folderId, environments: [environment] };
            const gen: SagaIterator = initApplicationForm({
                payload: { appId: applicationId, folderId },
                type: 'whatever',
            });
            expect(gen.next(applicationId).value).toStrictEqual(call(httpGET, `api/v1/applications/${applicationId}`));
            expect(gen.next({ data: application }).value).toStrictEqual(put(setApplication(application)));
            expect(gen.next(folderId).value).toStrictEqual(put(setApplicationFolder(folderId)));

            expect(gen.next().value).toStrictEqual(call(httpPOST, 'api/v1/environments/search', { title: '', folderId }));
            expect(gen.next({ data: environments }).value).toStrictEqual(put(setEnvironments(environments)));

            expect(gen.next().value).toStrictEqual(call(httpPOST, 'api/v1/environments/stages/search', { title: '' }));
            expect(gen.next({ data: [stage] }).value).toStrictEqual(put(setStages([stage])));

            expect(gen.next().value).toStrictEqual(call(httpPOST, 'api/v1/applications/search', { title: '', folderId }));
            expect(gen.next({ data: allApplications }).value).toStrictEqual(put(setApplications(allApplications)));

            expect(gen.next().done).toBe(true);
        });

        it('should load all the data needed on init while editing application on folder', () => {
            const applicationId = 'Application1';
            const folderId = 'Folder42';
            const application = { id: applicationId, title: 'App1', folderId, environments: [environment] };
            const gen: SagaIterator = initApplicationForm({
                payload: { appId: applicationId, folderId },
                type: 'whatever',
            });
            expect(gen.next(applicationId).value).toStrictEqual(call(httpGET, `api/v1/applications/${applicationId}`));
            expect(gen.next({ data: application }).value).toStrictEqual(put(setApplication(application)));
            expect(gen.next(folderId).value).toStrictEqual(put(setApplicationFolder(folderId)));

            expect(gen.next().value).toStrictEqual(call(httpPOST, 'api/v1/environments/search', { title: '', folderId }));
            expect(gen.next({ data: environments }).value).toStrictEqual(put(setEnvironments(environments)));

            expect(gen.next().value).toStrictEqual(call(httpPOST, 'api/v1/environments/stages/search', { title: '' }));
            expect(gen.next({ data: [stage] }).value).toStrictEqual(put(setStages([stage])));

            expect(gen.next().value).toStrictEqual(call(httpPOST, 'api/v1/applications/search', { title: '', folderId }));
            expect(gen.next({ data: allApplications }).value).toStrictEqual(put(setApplications(allApplications)));

            expect(gen.next().done).toBe(true);
        });

        it('should load all the data needed on init while creating application', () => {
            const applicationId = '';
            const folderId = '';
            const gen: SagaIterator = initApplicationForm({
                payload: { appId: applicationId, folderId },
                type: 'whatever',
            });
            expect(gen.next().value).toStrictEqual(call(httpPOST, 'api/v1/environments/search', { title: '', folderId }));
            expect(gen.next({ data: environments }).value).toStrictEqual(put(setEnvironments(environments)));

            expect(gen.next().value).toStrictEqual(call(httpPOST, 'api/v1/environments/stages/search', { title: '' }));
            expect(gen.next({ data: [stage] }).value).toStrictEqual(put(setStages([stage])));

            expect(gen.next().value).toStrictEqual(call(httpPOST, 'api/v1/applications/search', { title: '', folderId }));
            expect(gen.next({ data: allApplications }).value).toStrictEqual(put(setApplications(allApplications)));

            expect(gen.next().done).toBe(true);
        });

        it('should load all the data needed on init while creating application on folder', () => {
            const applicationId = '';
            const folderId = 'Folder42';
            const gen: SagaIterator = initApplicationForm({
                payload: { appId: applicationId, folderId },
                type: 'whatever',
            });

            expect(gen.next().value).toStrictEqual(call(httpPOST, 'api/v1/environments/search', { title: '', folderId }));
            expect(gen.next({ data: environments }).value).toStrictEqual(put(setEnvironments(environments)));

            expect(gen.next().value).toStrictEqual(call(httpPOST, 'api/v1/environments/stages/search', { title: '' }));
            expect(gen.next({ data: [stage] }).value).toStrictEqual(put(setStages([stage])));

            expect(gen.next().value).toStrictEqual(call(httpPOST, 'api/v1/applications/search', { title: '', folderId }));
            expect(gen.next({ data: allApplications }).value).toStrictEqual(put(setApplications(allApplications)));

            expect(gen.next().done).toBe(true);
        });
    });

    describe('cancelApplicationForm', () => {
        it('should cancel the form', () => {
            const $state = { go: jest.fn() };
            const folderId = '';
            const gen: SagaIterator = cancelApplicationForm({
                payload: folderId,
                type: 'whatever',
            });
            expect(gen.next().value).toStrictEqual(call(getAngularService, '$state'));
            expect(gen.next($state).value).toStrictEqual(call(removeConfirmLeavePopup));
            expect(gen.next().value).toStrictEqual(put(onFormClean));
            expect(gen.next(folderId).value).toStrictEqual($state.go('applicationList'));
            expect(gen.next().done).toBe(true);
        });

        it('should cancel the form inside folder', () => {
            const $state = { go: jest.fn() };
            const folderId = 'Folder42';
            const gen: SagaIterator = cancelApplicationForm({
                payload: folderId,
                type: 'whatever',
            });
            expect(gen.next().value).toStrictEqual(call(getAngularService, '$state'));
            expect(gen.next($state).value).toStrictEqual(call(removeConfirmLeavePopup));
            expect(gen.next().value).toStrictEqual(put(onFormClean));
            expect(gen.next(folderId).value).toStrictEqual($state.go('folders.detail.folderApplication'));
            expect(gen.next().done).toBe(true);
        });
    });

    describe('addConfirmLeavePopup', () => {
        it('should add confirmation popup', () => {
            const service = { requireConfirmation: jest.fn() };
            const gen: SagaIterator = addConfirmLeavePopup();
            expect(gen.next().value).toStrictEqual(call(getAngularService, 'ConfirmLeaveService'));
            gen.next(service);
            expect(gen.next().done).toBe(true);
        });

        it('should remove confirmation popup', () => {
            const service = { disableConfirmation: jest.fn() };
            const gen: SagaIterator = removeConfirmLeavePopup();
            expect(gen.next().value).toStrictEqual(call(getAngularService, 'ConfirmLeaveService'));
            gen.next(service);
            expect(gen.next().done).toBe(true);
        });
    });
});
