package com.xebialabs.xlrelease.environments.repository.sql.persistence.builder

import com.xebialabs.xlrelease.db.sql.SqlBuilder.Dialect
import com.xebialabs.xlrelease.db.sql.{Sql, SqlBuilder, SqlWithParameters}
import com.xebialabs.xlrelease.environments.repository.sql.persistence.schema.EnvironmentLabelSchema.ENV_LABELS
import com.xebialabs.xlrelease.environments.repository.sql.persistence.schema.EnvironmentSchema.{ENVIRONMENTS, ENV_TO_LABEL}
import com.xebialabs.xlrelease.environments.repository.sql.persistence.schema.EnvironmentStageSchema.ENV_STAGES
import com.xebialabs.xlrelease.repository.sql.persistence.CiId.CiId
import com.xebialabs.xlrelease.repository.sql.persistence.CiUid
import com.xebialabs.xlrelease.repository.sql.persistence.Schema.FOLDERS


object EnvironmentSqlBuilder {
  def apply(aliasPrefix: String = "")(implicit dialect: Dialect): EnvironmentSqlBuilder = {
    new EnvironmentSqlBuilder(aliasPrefix)
  }
}

class EnvironmentSqlBuilder(aliasPrefix: String)(implicit dialect: Dialect) extends SqlBuilder[EnvironmentSqlBuilder] with InFolderSqlBuilder {
  private val envToLabelAlias = makeAlias(aliasPrefix, "envToLabel")
  private val envLabelAlias = makeAlias(aliasPrefix, "envLabel")
  private val baseEnvAlias = makeAlias(aliasPrefix, "baseEnv")
  private val folderAlias = makeAlias(aliasPrefix, "folder")

  private lazy val baseBuilder = new BaseEnvironmentSqlBuilder(baseEnvAlias)

  def select(): EnvironmentSqlBuilder = this

  override def build(): SqlWithParameters = {
    val (innerQuery, innerParams) = baseBuilder.select().build()
    val stmt =
      s"""
         |SELECT
         | $baseEnvAlias.${ColumnAliases.Environments.CI_UID} ${ColumnAliases.Environments.CI_UID}
         | ,$baseEnvAlias.${ColumnAliases.Environments.FOLDER_ID} ${ColumnAliases.Environments.FOLDER_ID}
         | ,$baseEnvAlias.${ColumnAliases.Environments.FOLDER_PATH} ${ColumnAliases.Environments.FOLDER_PATH}
         | ,$baseEnvAlias.${ColumnAliases.Environments.ID} ${ColumnAliases.Environments.ID}
         | ,$baseEnvAlias.${ColumnAliases.Environments.TITLE} ${ColumnAliases.Environments.TITLE}
         | ,$baseEnvAlias.${ColumnAliases.Environments.DESCRIPTION} ${ColumnAliases.Environments.DESCRIPTION}
         | ,$baseEnvAlias.${ColumnAliases.EnvStages.ID} ${ColumnAliases.EnvStages.ID}
         | ,$baseEnvAlias.${ColumnAliases.EnvStages.TITLE} ${ColumnAliases.EnvStages.TITLE}
         | ,$envLabelAlias.${ENV_LABELS.ID} ${ColumnAliases.EnvLabels.ID}
         | ,$envLabelAlias.${ENV_LABELS.TITLE} ${ColumnAliases.EnvLabels.TITLE}
         | ,$envLabelAlias.${ENV_LABELS.COLOR} ${ColumnAliases.EnvLabels.COLOR}
         | FROM (${indent(innerQuery)}) $baseEnvAlias
         | LEFT JOIN ${ENV_TO_LABEL.TABLE} $envToLabelAlias
         |  ON $baseEnvAlias.${ColumnAliases.Environments.CI_UID} = $envToLabelAlias.${ENV_TO_LABEL.ENVIRONMENT_UID}
         | LEFT JOIN ${ENV_LABELS.TABLE} $envLabelAlias
         |  ON $envToLabelAlias.${ENV_TO_LABEL.LABEL_UID} = $envLabelAlias.${ENV_LABELS.CI_UID}
      """.stripMargin
    super.select(stmt)
    super.orderBy(ColumnAliases.Environments.CI_UID)
    val fullSql = super.build()
    fullSql._1 -> (innerParams ++ fullSql._2)
  }

  override def withFolder(folderId: CiUid): EnvironmentSqlBuilder = {
    baseBuilder.withFolderId(folderId)
    this
  }

  override def withGlobalAndFolder(folderId: CiUid): EnvironmentSqlBuilder = {
    baseBuilder.withGlobalAndFolder(folderId)
    this
  }

  override def withNoFolder(): EnvironmentSqlBuilder = {
    baseBuilder.withNoFolder()
    this
  }

  def withTitleLike(title: String): EnvironmentSqlBuilder = {
    baseBuilder.withTitleLike(title)
    this
  }

  def withStageTitle(stageTitle: String): EnvironmentSqlBuilder = {
    baseBuilder.withStageTitle(stageTitle)
    this
  }

  def withStageTitles(titles: Iterable[String]): EnvironmentSqlBuilder = {
    baseBuilder.withStageTitles(titles)
    this
  }

  def withLabelTitles(titles: Iterable[String]): EnvironmentSqlBuilder = {
    baseBuilder.withLabelTitles(titles)
    this
  }

  def withEnvironmentId(id: CiId): EnvironmentSqlBuilder = {
    baseBuilder.withEnvironmentId(id)
    this
  }

  def withEnvironmentTitle(title: String): EnvironmentSqlBuilder = {
    baseBuilder.withEnvironmentTitle(title)
    this
  }

  override def limitAndOffset(limit: Long, offset: Long): EnvironmentSqlBuilder = {
    baseBuilder.limitAndOffset(limit, offset)
    this
  }

  override def orderBy(column: String): EnvironmentSqlBuilder = {
    baseBuilder.orderBy(column)
    this
  }

  override def newInstance: EnvironmentSqlBuilder = new EnvironmentSqlBuilder(aliasPrefix)

  class BaseEnvironmentSqlBuilder(aliasPrefix: String)(implicit dialect: Dialect) extends SqlBuilder[BaseEnvironmentSqlBuilder] {
    private val envAlias = makeAlias(aliasPrefix, "env")
    private val stageAlias = makeAlias(aliasPrefix, "stage")
    private val envToLabelAlias = makeAlias(aliasPrefix, "envToLabel")
    private val labelAlias = makeAlias(aliasPrefix, "label")

    def select(): BaseEnvironmentSqlBuilder = this

    // Many-to-many joins are not returned, they are only used for filtering
    // This is to make filtering work properly
    override def build(): SqlWithParameters = {
      val stmt =
        s"""
           |SELECT DISTINCT
           | $envAlias.${ENVIRONMENTS.CI_UID} ${ColumnAliases.Environments.CI_UID}
           | ,$folderAlias.${FOLDERS.FOLDER_ID} ${ColumnAliases.Environments.FOLDER_ID}
           | ,$folderAlias.${FOLDERS.FOLDER_PATH} ${ColumnAliases.Environments.FOLDER_PATH}
           | ,$envAlias.${ENVIRONMENTS.ID} ${ColumnAliases.Environments.ID}
           | ,$envAlias.${ENVIRONMENTS.TITLE} ${ColumnAliases.Environments.TITLE}
           | ,$envAlias.${ENVIRONMENTS.DESCRIPTION} ${ColumnAliases.Environments.DESCRIPTION}
           | ,$envAlias.${ENVIRONMENTS.ENV_STAGE_UID} ${ColumnAliases.Environments.ENV_STAGE_UID}
           | ,$stageAlias.${ENV_STAGES.CI_UID} ${ColumnAliases.EnvStages.CI_UID}
           | ,$stageAlias.${ENV_STAGES.ID} ${ColumnAliases.EnvStages.ID}
           | ,$stageAlias.${ENV_STAGES.TITLE} ${ColumnAliases.EnvStages.TITLE}
           | FROM ${ENVIRONMENTS.TABLE} $envAlias
           | INNER JOIN ${ENV_STAGES.TABLE} $stageAlias
           |  ON $envAlias.${ENVIRONMENTS.ENV_STAGE_UID} = $stageAlias.${ENV_STAGES.CI_UID}
           | LEFT JOIN ${ENV_TO_LABEL.TABLE} $envToLabelAlias
           |  ON $envAlias.${ENVIRONMENTS.CI_UID} = $envToLabelAlias.${ENV_TO_LABEL.ENVIRONMENT_UID}
           | LEFT JOIN ${ENV_LABELS.TABLE} $labelAlias
           |  ON $envToLabelAlias.${ENV_TO_LABEL.LABEL_UID} = $labelAlias.${ENV_LABELS.CI_UID}
           | LEFT JOIN ${FOLDERS.TABLE} $folderAlias
           |  ON $envAlias.${ENVIRONMENTS.FOLDER_UID} = $folderAlias.${FOLDERS.CI_UID}
       """.stripMargin
      super.select(stmt)

      super.build()
    }

    def withFolderId(id: CiUid): BaseEnvironmentSqlBuilder = {
      conditions += Sql(s"$envAlias.${ENVIRONMENTS.FOLDER_UID} = ?", Seq(id))
      this
    }

    def withGlobalAndFolder(id: CiUid): BaseEnvironmentSqlBuilder = {
      conditions += Sql(s"$envAlias.${ENVIRONMENTS.FOLDER_UID} = ? OR $envAlias.${ENVIRONMENTS.FOLDER_UID} IS NULL", Seq(id))
      this
    }

    def withNoFolder(): BaseEnvironmentSqlBuilder = {
      conditions += Sql(s"$envAlias.${ENVIRONMENTS.FOLDER_UID} IS NULL", Seq.empty)
      this
    }

    def withTitleLike(title: String): BaseEnvironmentSqlBuilder = {
      like(s"$envAlias.${ENVIRONMENTS.TITLE}", title)
    }

    def withStageTitle(stageTitle: String): BaseEnvironmentSqlBuilder = {
      conditions += Sql(s"$stageAlias.${ENV_STAGES.TITLE} = ?", Seq(stageTitle))
      this
    }

    def withStageTitles(titles: Iterable[String]): BaseEnvironmentSqlBuilder = {
      conditions ++= whereInCondition(s"$stageAlias.${ENV_STAGES.TITLE}", titles).toList
      this
    }

    def withEnvironmentId(id: CiId): BaseEnvironmentSqlBuilder = {
      conditions += Sql(s"$envAlias.${ENVIRONMENTS.ID} = ?", Seq(id))
      this
    }

    def withEnvironmentTitle(title: CiId): BaseEnvironmentSqlBuilder = {
      conditions += Sql(s"LOWER($envAlias.${ENVIRONMENTS.TITLE}) = LOWER(?)", Seq(title))
      this
    }

    def withLabelTitles(titles: Iterable[String]): BaseEnvironmentSqlBuilder = {
      conditions ++= whereInCondition(s"LOWER($labelAlias.${ENV_LABELS.TITLE})", titles, s => s"LOWER($s)").toList
      this
    }

    override def newInstance: BaseEnvironmentSqlBuilder = new BaseEnvironmentSqlBuilder(aliasPrefix)

  }

}
