import React, { useEffect, useState } from 'react';
import { itemAlreadyDefined } from '../../../../../../../../../../core/xlr-ui/app/react/utils/validators';
import { Stage } from '../../../application-list.types';
import { DotConfirmationDialog, DotInputText, DotThemeProvider } from '@digital-ai/dot-components';
import { useAppSelector } from '../../../../../../../../../../core/xlr-ui/app/js/hooks';
import { getStageList } from '../ducks/environment-list.selectors';

const nameOptions = {
    label: 'Name',
    description: 'Name of the stage',
    required: true,
};

const initStage = { id: '', title: '' } as Stage;

interface StageEditProps {
    isOpened: boolean;
    model: Stage;
    onCancel: () => void;
    onSave: (st: Stage) => void;
}

export const StageEditModal = ({ isOpened, model, onCancel, onSave }: StageEditProps) => {
    const [localStage, setLocalStage] = useState(model);
    const [valid, setValid] = useState(true);
    const [disabledButton, setDissabledButton] = useState(true);
    const stages = useAppSelector(getStageList);

    useEffect(() => {
        setLocalStage(model);
    }, [model]);
    const handleButtonClick = (action: string) => {
        switch (action) {
            case 'update':
                setLocalStage({ ...localStage, id: model.id });
                onSave(localStage);
                break;
            case 'cancel':
                onCancel();
                break;
            default:
                throw 'Unknown action';
        }
        resetForm();
    };

    const handleModelChange = (newTitle: string) => {
        if (newTitle !== localStage.title) {
            if (itemAlreadyDefined(stages, model.id, 'title', newTitle)) {
                setValid(false);
                setDissabledButton(true);
                return;
            }
            setDissabledButton(false);
            setLocalStage({
                ...localStage,
                title: newTitle,
            });
            setValid(true);
        } else {
            setDissabledButton(true);
        }
    };

    const resetForm = () => {
        setLocalStage(initStage);
    };

    const modalHeader = 'Edit stage';

    return (
        <DotThemeProvider>
            <DotConfirmationDialog
                className="xl-default-modal"
                message={
                    <DotInputText
                        autoFocus={true}
                        defaultValue={model.title}
                        error={!valid}
                        helperText={valid ? '' : 'Name of stage already exists.'}
                        id={'stage-modal-title'}
                        label={nameOptions.label}
                        maxLength={255}
                        name={'stage-modal-title'}
                        onChange={(e) => handleModelChange(e.target.value)}
                        persistentLabel={true}
                        required={nameOptions.required}
                    />
                }
                onCancel={() => handleButtonClick('cancel')}
                onSubmit={() => handleButtonClick('update')}
                open={isOpened}
                submitButtonProps={{
                    className: 'submitBtn',
                    disabled: disabledButton,
                    label: 'Save',
                }}
                title={modalHeader}
            />
        </DotThemeProvider>
    );
};
