package com.xebialabs.xlrelease.environments.repository.sql.persistence

import com.xebialabs.deployit.exception.NotFoundException
import com.xebialabs.xlrelease.db.sql.SqlBuilder.Dialect
import com.xebialabs.xlrelease.db.sql.SqlWithParameters
import com.xebialabs.xlrelease.db.sql.transaction.IsTransactional
import com.xebialabs.xlrelease.domain.environments.{DeploymentTarget, Environment}
import com.xebialabs.xlrelease.environments.repository.sql.persistence.EnvironmentPersistence.{copyEnvironment, prepareIdsForSerialization}
import com.xebialabs.xlrelease.environments.repository.sql.persistence.data.EnvironmentRow.{environmentListResultSetExtractor, environmentResultSetExtractor}
import com.xebialabs.xlrelease.environments.repository.sql.persistence.data.EnvironmentSearchResultRow.environmentSearchResultSetExtractor
import com.xebialabs.xlrelease.environments.repository.sql.persistence.data.{EnvironmentRow, EnvironmentSearchResultRow}
import com.xebialabs.xlrelease.environments.repository.sql.persistence.schema.EnvironmentSchema
import com.xebialabs.xlrelease.environments.repository.sql.persistence.schema.EnvironmentSchema.{ENVIRONMENTS, ENV_TO_LABEL}
import com.xebialabs.xlrelease.repository.CiCloneHelper.{cloneCi, cloneCis}
import com.xebialabs.xlrelease.repository.sql.persistence.CiId.CiId
import com.xebialabs.xlrelease.repository.sql.persistence.Schema.FOLDERS
import com.xebialabs.xlrelease.repository.sql.persistence.Utils._
import com.xebialabs.xlrelease.repository.sql.persistence.{CiUid, FolderPersistence, PersistenceSupport, SecurablePersistence}
import com.xebialabs.xlrelease.serialization.json.utils.CiSerializerHelper.serialize
import com.xebialabs.xlrelease.service.CiIdService
import org.springframework.beans.factory.annotation.{Autowired, Qualifier}
import org.springframework.dao.DuplicateKeyException
import org.springframework.jdbc.core.JdbcTemplate
import org.springframework.stereotype.Repository
import org.springframework.util.CollectionUtils

import java.util.Collections
import scala.jdk.CollectionConverters._
import scala.util.Try

object EnvironmentPersistence {
  def prepareIdsForSerialization(env: Environment)(implicit ciIdService: CiIdService): Unit = {
    prepareId(env, () => createPersistedId[Environment])
    Option(env.getDeploymentTarget).foreach(t => {
      prepareId(t, () => createPersistedId[DeploymentTarget])
    })
    Option(env.getStage).foreach(s => s.setId(toPersistedId(s.getId)))
    Option(env.getLabels).getOrElse(Collections.emptyList()).asScala.foreach(l => l.setId(toPersistedId(l.getId)))
  }

  def copyEnvironment(env: Environment): Environment = {
    val environmentCopy = cloneCi(env)
    Option(env.getDeploymentTarget).foreach(t => environmentCopy.setDeploymentTarget(cloneCi(t)))
    Option(env.getStage).foreach(s => environmentCopy.setStage(cloneCi(s)))
    environmentCopy.setLabels(cloneCis(env.getLabels))
    environmentCopy
  }

  def copyEnvironments(envs: List[Environment]): List[Environment] = {
    envs.map(copyEnvironment)
  }
}

@Repository
@IsTransactional
class EnvironmentPersistence @Autowired()(@Qualifier("xlrRepositoryJdbcTemplate") implicit val jdbcTemplate: JdbcTemplate,
                                          @Qualifier("xlrRepositorySqlDialect") implicit val dialect: Dialect,
                                          implicit val ciIdService: CiIdService,
                                          val folderPersistence: FolderPersistence,
                                          val securablePersistence: SecurablePersistence,
                                          val environmentStagePersistence: EnvironmentStagePersistence,
                                          val environmentLabelPersistence: EnvironmentLabelPersistence)
  extends PersistenceSupport {

  private val STMT_INSERT_ENVIRONMENTS =
    s"""
       |INSERT INTO ${ENVIRONMENTS.TABLE}
       |(${ENVIRONMENTS.CI_UID},
       |${ENVIRONMENTS.FOLDER_UID},
       |${ENVIRONMENTS.ID},
       |${ENVIRONMENTS.CORRELATION_UID},
       |${ENVIRONMENTS.TITLE},
       |${ENVIRONMENTS.DESCRIPTION},
       |${ENVIRONMENTS.ENV_STAGE_UID},
       |${ENVIRONMENTS.CONTENT})
       |VALUES
       |(:${ENVIRONMENTS.CI_UID},
       |:${ENVIRONMENTS.FOLDER_UID},
       |:${ENVIRONMENTS.ID},
       |:${ENVIRONMENTS.CORRELATION_UID},
       |:${ENVIRONMENTS.TITLE},
       |:${ENVIRONMENTS.DESCRIPTION},
       |:${ENVIRONMENTS.ENV_STAGE_UID},
       |:${ENVIRONMENTS.CONTENT})
      """.stripMargin

  private val STMT_EXISTS_BY_TITLE_IGNORECASE_GLOBAL =
    s"""|SELECT COUNT(*) FROM ${ENVIRONMENTS.TABLE} WHERE
        |  LOWER(${ENVIRONMENTS.TITLE}) = LOWER(:${ENVIRONMENTS.TITLE})
        |  AND ${ENVIRONMENTS.FOLDER_UID} IS NULL""".stripMargin

  private val STMT_EXISTS_BY_TITLE_IGNORECASE_FOLDER =
    s"""|SELECT COUNT(*) FROM ${ENVIRONMENTS.TABLE} WHERE
        |  LOWER(${ENVIRONMENTS.TITLE}) = LOWER(:${ENVIRONMENTS.TITLE})
        |  AND ${ENVIRONMENTS.FOLDER_UID} = :${ENVIRONMENTS.FOLDER_UID}""".stripMargin

  def insert(environment: Environment): CiId = {
    sanitizeEnvironmentInput(environment)

    val stageUid = environmentStagePersistence.findUidById(environment.getStage.getId)
      .getOrElse(throw new NotFoundException(s"Environment stage [${environment.getStage.getId}] not found"))

    val exists = Option(environment.getFolderId).map(id =>
      sqlQuery(
        STMT_EXISTS_BY_TITLE_IGNORECASE_FOLDER,
        params(ENVIRONMENTS.TITLE -> environment.getTitle, ENVIRONMENTS.FOLDER_UID -> folderPersistence.getUid(id)),
        _.getInt(1) > 0
      ).head
    ).getOrElse(
        sqlQuery(STMT_EXISTS_BY_TITLE_IGNORECASE_GLOBAL, params(ENVIRONMENTS.TITLE -> environment.getTitle), _.getInt(1) > 0).head
    )
    if (exists) {
      throw new IllegalArgumentException(alreadyExistsMessage("Environment", environment.getTitle, Option(environment.getFolderId)))
    }

    val labelIdsToUids = getLabelUids(environment)

    val ciUid = securablePersistence.insert()
    val environmentCopy = copyEnvironment(environment)
    prepareIdsForSerialization(environmentCopy)
    val content = serialize(environmentCopy)
    val displayedEnvId = toDisplayId(environmentCopy.getId)
    try {
      sqlExecWithContent(
        STMT_INSERT_ENVIRONMENTS,
        params(
          ENVIRONMENTS.CI_UID -> ciUid,
          ENVIRONMENTS.FOLDER_UID -> Option(environmentCopy.getFolderId).map(folderPersistence.getUid).orNull,
          ENVIRONMENTS.ID -> environmentCopy.getId,
          ENVIRONMENTS.CORRELATION_UID -> environmentCopy.getCorrelationUid,
          ENVIRONMENTS.DESCRIPTION -> environmentCopy.getDescription,
          ENVIRONMENTS.TITLE -> environmentCopy.getTitle,
          ENVIRONMENTS.ENV_STAGE_UID -> stageUid),
        ENVIRONMENTS.CONTENT -> content,
        identity
      )
    } catch {
      case ex: DuplicateKeyException =>
        throw new IllegalArgumentException(s"Environment with ID '$displayedEnvId' already exists", ex)
    }

    insertEnvToLabelReferences(ciUid, labelIdsToUids.valuesIterator.toSet)

    displayedEnvId
  }

  private val STMT_UPDATE_ENVIRONMENTS =
    s"""|UPDATE ${ENVIRONMENTS.TABLE}
        | SET
        |  ${ENVIRONMENTS.DESCRIPTION} = :${ENVIRONMENTS.DESCRIPTION},
        |  ${ENVIRONMENTS.TITLE} = :${ENVIRONMENTS.TITLE},
        |  ${ENVIRONMENTS.ENV_STAGE_UID} = :${ENVIRONMENTS.ENV_STAGE_UID},
        |  ${ENVIRONMENTS.CONTENT} = :${ENVIRONMENTS.CONTENT}
        | WHERE
        |  ${ENVIRONMENTS.CI_UID} = :${ENVIRONMENTS.CI_UID}
        """.stripMargin

  private val STMT_EXISTS_ANOTHER_ENV_WITH_TITLE_GLOBAL =
    s"""
       |$STMT_EXISTS_BY_TITLE_IGNORECASE_GLOBAL
       |AND ${ENVIRONMENTS.CI_UID} <> :${ENVIRONMENTS.CI_UID}
      """.stripMargin

  private val STMT_EXISTS_ANOTHER_ENV_WITH_TITLE_FOLDER =
    s"""
       |$STMT_EXISTS_BY_TITLE_IGNORECASE_FOLDER
       |AND ${ENVIRONMENTS.CI_UID} <> :${ENVIRONMENTS.CI_UID}
      """.stripMargin

  def update(environment: Environment): Unit = {
    val environmentUid = findUidById(environment.getId)
      .getOrElse(throw new NotFoundException(s"Environment [${environment.getId}] not found"))
    val stageUid = environmentStagePersistence.findUidById(environment.getStage.getId)
      .getOrElse(throw new NotFoundException(s"Environment stage [${environment.getStage.getId}] not found"))

    sanitizeEnvironmentInput(environment)

    val existsAnotherWithSameTitle = Option(environment.getFolderId).map(id =>
      sqlQuery(
        STMT_EXISTS_ANOTHER_ENV_WITH_TITLE_FOLDER,
        params(ENVIRONMENTS.TITLE -> environment.getTitle, ENVIRONMENTS.CI_UID -> environmentUid, ENVIRONMENTS.FOLDER_UID -> folderPersistence.getUid(id)),
        _.getInt(1) > 0
      ).head
    ).getOrElse(
      sqlQuery(
        STMT_EXISTS_ANOTHER_ENV_WITH_TITLE_GLOBAL,
        params(ENVIRONMENTS.TITLE -> environment.getTitle, ENVIRONMENTS.CI_UID -> environmentUid),
        _.getInt(1) > 0
      ).head
    )

    if (existsAnotherWithSameTitle) {
      throw new IllegalArgumentException(alreadyExistsMessage("Environment", environment.getTitle, Option(environment.getFolderId)))
    }

    updateEnvToLabelReferences(environment, environmentUid)

    val environmentCopy = copyEnvironment(environment)
    prepareIdsForSerialization(environmentCopy)

    sqlExecWithContent(
      STMT_UPDATE_ENVIRONMENTS,
      params(
        ENVIRONMENTS.DESCRIPTION -> environmentCopy.getDescription,
        ENVIRONMENTS.TITLE -> environmentCopy.getTitle,
        ENVIRONMENTS.CI_UID -> environmentUid,
        ENVIRONMENTS.ENV_STAGE_UID -> stageUid,
      ),
      ENVIRONMENTS.CONTENT -> serialize(environmentCopy), checkCiUpdated(environmentCopy.getId)
    )
  }

  def findIdsByQuery(sqlWithParameters: SqlWithParameters): Seq[CiId] = {
    val (sql, params) = sqlWithParameters
    jdbcTemplate.queryForList[CiId](sql, classOf[CiId], params: _*).asScala.toSeq
  }

  private val BASE_STMT_SELECT_ENVIRONMENT_ROW =
    s"""|SELECT env.${ENVIRONMENTS.CI_UID} ${ENVIRONMENTS.CI_UID},
        |env.${ENVIRONMENTS.ID} ${ENVIRONMENTS.ID},
        |env.${ENVIRONMENTS.CONTENT} ${ENVIRONMENTS.CONTENT},
        |folder.${FOLDERS.FOLDER_ID} ${FOLDERS.FOLDER_ID},
        |folder.${FOLDERS.FOLDER_PATH} ${FOLDERS.FOLDER_PATH}
        |FROM ${ENVIRONMENTS.TABLE} env
        |LEFT JOIN ${FOLDERS.TABLE} folder
        | ON env.${ENVIRONMENTS.FOLDER_UID} = folder.${FOLDERS.CI_UID}""".stripMargin


  def findById(environmentId: CiId): Option[EnvironmentRow] = {
    val stmt =
      s"""|$BASE_STMT_SELECT_ENVIRONMENT_ROW
          |WHERE env.${ENVIRONMENTS.ID} = :${ENVIRONMENTS.ID}""".stripMargin
    sqlQuery(stmt, params(ENVIRONMENTS.ID -> toPersistedId(environmentId)), environmentResultSetExtractor())
  }

  def findByTitle(environmentTitle: CiId): Option[EnvironmentRow] = {
    val stmt =
      s"""|$BASE_STMT_SELECT_ENVIRONMENT_ROW
          |WHERE LOWER(env.${ENVIRONMENTS.TITLE}) = LOWER(${ENVIRONMENTS.TITLE})""".stripMargin
    sqlQuery(stmt, params(ENVIRONMENTS.TITLE -> environmentTitle), environmentResultSetExtractor())
  }

  def findInFolderByCorrelationId(folderId: String, correlationId: String): Option[EnvironmentRow] = {

    val stmt =
      s"""|$BASE_STMT_SELECT_ENVIRONMENT_ROW
          |WHERE env.${ENVIRONMENTS.CORRELATION_UID} = :${ENVIRONMENTS.CORRELATION_UID}
          | AND env.${ENVIRONMENTS.FOLDER_UID} = :${ENVIRONMENTS.FOLDER_UID}""".stripMargin
    sqlQuery(stmt,
      params(
        ENVIRONMENTS.CORRELATION_UID -> correlationId,
        ENVIRONMENTS.FOLDER_UID -> folderPersistence.getUid(folderId)
      ),
      environmentResultSetExtractor())
  }

  def findUidById(environmentId: CiId): Option[CiUid] = {
    val stmt =
      s"""|SELECT ${ENVIRONMENTS.CI_UID}
          |FROM ${ENVIRONMENTS.TABLE}
          |WHERE ${ENVIRONMENTS.ID} = :environmentId""".stripMargin
    sqlQuery(stmt, params("environmentId" -> toPersistedId(environmentId)), rs => CiUid(rs.getInt(ENVIRONMENTS.CI_UID))).headOption
  }

  def search(sqlWithParameters: SqlWithParameters): Vector[EnvironmentSearchResultRow] = {
    val (sql, params) = sqlWithParameters
    jdbcTemplate.query(sql, environmentSearchResultSetExtractor(), params: _*)
  }

  def fetchEnvironments(environmentIds: List[CiId]): Seq[EnvironmentRow] = {
      val stmt =
        s"""|$BASE_STMT_SELECT_ENVIRONMENT_ROW
            |WHERE env.${ENVIRONMENTS.ID} IN (:environmentIds)""".stripMargin
    sqlQuery(stmt,
      params(
        "environmentIds" -> environmentIds.map(toPersistedId).asJava
      ),
      environmentListResultSetExtractor())
  }

  def fetchEnvironmentsByFolderId(folderId: String): Seq[EnvironmentRow] = {
    val stmt =
      s"""|$BASE_STMT_SELECT_ENVIRONMENT_ROW
          |WHERE env.${ENVIRONMENTS.FOLDER_UID} = :${ENVIRONMENTS.FOLDER_UID}""".stripMargin
    sqlQuery(stmt,
      params(
        ENVIRONMENTS.FOLDER_UID -> folderPersistence.getUid(folderId)
      ),
      environmentListResultSetExtractor())
  }

  private val STMT_DELETE_ENVIRONMENTS_BY_ID =
    s"""|DELETE FROM ${ENVIRONMENTS.TABLE}
        | WHERE ${ENVIRONMENTS.ID} = :environmentId""".stripMargin

  def delete(environmentId: CiId): Try[Boolean] =
    sqlExec(STMT_DELETE_ENVIRONMENTS_BY_ID, params("environmentId" -> toPersistedId(environmentId)), ps => Try(ps.execute()))

  private def getLabelUids(environment: Environment): Map[CiId, CiUid] = {
    if (CollectionUtils.isEmpty(environment.getLabels)) {
      Map.empty[CiId, CiUid]
    } else {
      val labelIds = environment.getLabels.asScala.map(_.getId).toSet
      val labelIdsToUids = environmentLabelPersistence.getUidsByIds(labelIds)
      val labelDiff = labelIds.map(toDisplayId).diff(labelIdsToUids.keySet)
      if (labelDiff.nonEmpty) {
        throw new NotFoundException(s"Environment labels [${labelDiff.mkString(", ")}] not found")
      }
      labelIdsToUids
    }
  }

  private def insertEnvToLabelReferences(environmentUid: CiUid, labelUids: Set[CiUid]) = {
    val insertEnvToLabelStmt =
      s"""INSERT INTO ${ENV_TO_LABEL.TABLE} (
         |${ENV_TO_LABEL.ENVIRONMENT_UID},
         |${ENV_TO_LABEL.LABEL_UID}
         |)
         |VALUES (
         |:${ENV_TO_LABEL.ENVIRONMENT_UID},
         |:${ENV_TO_LABEL.LABEL_UID}
         |)
       """.stripMargin

    sqlBatch(insertEnvToLabelStmt,
      labelUids.map { labelUid =>
        params(
          ENV_TO_LABEL.ENVIRONMENT_UID -> environmentUid,
          ENV_TO_LABEL.LABEL_UID -> labelUid
        )
      }
    )
  }

  private def updateEnvToLabelReferences(environment: Environment, environmentUid: CiUid) = {
    val labelIdsToUids = getLabelUids(environment)

    val deleteEnvToLabelRefsStmt =
      s"""
         |DELETE FROM ${ENV_TO_LABEL.TABLE}
         | WHERE
         | ${ENV_TO_LABEL.ENVIRONMENT_UID} = :${ENV_TO_LABEL.ENVIRONMENT_UID}
       """.stripMargin
    sqlExec(deleteEnvToLabelRefsStmt, params(ENV_TO_LABEL.ENVIRONMENT_UID -> environmentUid), _.execute())
    insertEnvToLabelReferences(environmentUid, labelIdsToUids.valuesIterator.toSet)
  }

  def getUidsByIds(environmentIds: Iterable[CiId]): Map[CiId, CiUid] = {
    val stmt =
      s"""|SELECT ${ENVIRONMENTS.CI_UID}, ${ENVIRONMENTS.ID}
          |FROM ${ENVIRONMENTS.TABLE}
          |WHERE ${ENVIRONMENTS.ID} IN (:environmentIds)
       """.stripMargin
    sqlQuery(stmt,
      params("environmentIds" -> environmentIds.map(toPersistedId).asJava),
      rs => rs.getCiId(ENVIRONMENTS.ID) -> CiUid(rs.getInt(ENVIRONMENTS.CI_UID))
    ).toMap
  }

  private def sanitizeEnvironmentInput(environment: Environment): Unit = {
    environment.setTitle(environment.getTitle.trimAndTruncate(EnvironmentSchema.TITLE_LENGTH))
    environment.setDescription(environment.getDescription.trimAndTruncate(EnvironmentSchema.DESCRIPTION_LENGTH))
  }
}
