package com.xebialabs.xlrelease.environments.repository.sql

import com.xebialabs.deployit.exception.NotFoundException
import com.xebialabs.xlrelease.api.v1.filter.EnvironmentLabelFilters
import com.xebialabs.xlrelease.db.sql.SqlBuilder.Dialect
import com.xebialabs.xlrelease.db.sql.transaction.{IsReadOnly, IsTransactional}
import com.xebialabs.xlrelease.domain.environments.EnvironmentLabel
import com.xebialabs.xlrelease.environments.repository.EnvironmentLabelRepository
import com.xebialabs.xlrelease.environments.repository.sql.persistence.EnvironmentLabelPersistence
import com.xebialabs.xlrelease.environments.repository.sql.persistence.builder.{ColumnAliases, EnvironmentLabelSqlBuilder}
import com.xebialabs.xlrelease.repository.Page
import com.xebialabs.xlrelease.service.CiIdService
import grizzled.slf4j.Logging
import org.springframework.beans.factory.annotation.{Autowired, Qualifier}
import org.springframework.stereotype.Repository

@IsTransactional
@Repository
class SqlEnvironmentLabelRepository @Autowired()(@Qualifier("xlrRepositorySqlDialect") implicit val dialect: Dialect,
                                                 implicit val ciIdService: CiIdService,
                                                 environmentLabelPersistence: EnvironmentLabelPersistence)
  extends EnvironmentLabelRepository with Logging {

  @IsReadOnly
  override def search(filters: EnvironmentLabelFilters, page: Page): Seq[EnvironmentLabel] = {
    val query = new EnvironmentLabelSqlBuilder()
      .select()
      .orderBy(ColumnAliases.EnvLabels.ID)
      .limitAndOffset(page.resultsPerPage, page.offset)

    if (filters != null && filters.getTitle != null) {
      query.withTitleLike(filters.getTitle)
    }

    environmentLabelPersistence.search(query.build())
  }

  @IsReadOnly
  override def findById(environmentLabelId: String): EnvironmentLabel = {
    environmentLabelPersistence.findById(environmentLabelId)
      .getOrElse(throw new NotFoundException(s"Environment label [$environmentLabelId] not found"))
  }

  @IsReadOnly
  override def findByTitle(environmentLabelTitle: String): EnvironmentLabel = {
    environmentLabelPersistence.findByTitle(environmentLabelTitle)
      .getOrElse(throw new NotFoundException(s"Environment label [$environmentLabelTitle] not found"))
  }

  override def create(environmentLabel: EnvironmentLabel): EnvironmentLabel = {
    environmentLabelPersistence.insert(environmentLabel)
  }

  override def update(environmentLabel: EnvironmentLabel): EnvironmentLabel = {
    environmentLabelPersistence.update(environmentLabel)
    findById(environmentLabel.getId)
  }

  override def delete(environmentLabelId: String): Unit = {
    findById(environmentLabelId)
    environmentLabelPersistence.delete(environmentLabelId)
  }
}
