package com.xebialabs.xlrelease.api.v1.impl;

import java.util.Date;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Controller;

import com.xebialabs.xlrelease.api.v1.EnvironmentReservationApi;
import com.xebialabs.xlrelease.api.v1.filter.ReservationFilters;
import com.xebialabs.xlrelease.api.v1.form.EnvironmentReservationForm;
import com.xebialabs.xlrelease.api.v1.views.EnvironmentReservationView;
import com.xebialabs.xlrelease.api.v1.views.search.EnvironmentReservationSearchView;
import com.xebialabs.xlrelease.domain.environments.Application;
import com.xebialabs.xlrelease.domain.environments.EnvironmentReservation;
import com.xebialabs.xlrelease.environments.service.ApplicationService;
import com.xebialabs.xlrelease.environments.service.EnvironmentReservationSecurity;
import com.xebialabs.xlrelease.environments.service.EnvironmentReservationService;

import io.micrometer.core.annotation.Timed;

import static com.xebialabs.xlrelease.api.v1.views.EnvironmentReservationView.fromEnvironmentReservation;

@Controller
public class EnvironmentReservationApiImpl implements EnvironmentReservationApi {

    private EnvironmentReservationService environmentReservationService;
    private EnvironmentReservationSecurity environmentReservationSecurity;
    private ApplicationService applicationService;

    @Autowired
    public EnvironmentReservationApiImpl(EnvironmentReservationService environmentReservationService, EnvironmentReservationSecurity environmentReservationSecurity, ApplicationService applicationService) {
        this.environmentReservationService = environmentReservationService;
        this.environmentReservationSecurity = environmentReservationSecurity;
        this.applicationService = applicationService;
    }

    @Timed
    @Override
    public EnvironmentReservationView createReservation(EnvironmentReservationForm environmentReservationForm) {
        environmentReservationSecurity.checkCreate();
        EnvironmentReservation reservation = environmentReservationForm.toEnvironmentReservation();
        return fromEnvironmentReservation(environmentReservationService.createReservation(reservation));
    }

    @Timed
    @Override
    public EnvironmentReservationView getReservation(String environmentReservationId) {
        environmentReservationSecurity.checkView();
        return fromEnvironmentReservation(environmentReservationService.findReservationById(environmentReservationId));
    }

    @Timed
    @Override
    public EnvironmentReservationView updateReservation(String environmentReservationId,
                                                        EnvironmentReservationForm environmentReservationForm) {
        environmentReservationSecurity.checkEdit();
        EnvironmentReservation reservation = environmentReservationForm.toEnvironmentReservation();
        reservation.setId(environmentReservationId);
        return fromEnvironmentReservation(environmentReservationService.updateReservation(reservation));
    }

    @Timed
    @Override
    public List<EnvironmentReservationSearchView> searchReservations(ReservationFilters filters) {
        environmentReservationSecurity.checkView();
        filters.validate();

        return environmentReservationService.searchReservations(filters)
                .entrySet()
                .stream()
                .map(e -> EnvironmentReservationSearchView.from(e.getKey(), e.getValue()))
                .collect(Collectors.toList());
    }

    @Timed
    @Override
    public void delete(String environmentReservationId) {
        environmentReservationSecurity.checkEdit();
        environmentReservationService.deleteReservation(environmentReservationId);
    }

    @Timed
    @Override
    public void addApplication(String environmentReservationId, String applicationId) {
        environmentReservationSecurity.checkEdit();
        EnvironmentReservation environmentReservation = environmentReservationService.findReservationById(environmentReservationId);
        Application application = applicationService.findApplicationById(applicationId);
        environmentReservation.addApplication(application);
        environmentReservationService.updateReservation(environmentReservation);
    }

    @Timed
    @Override
    public EnvironmentReservation create(EnvironmentReservation environmentReservation) {
        environmentReservationSecurity.checkCreate();
        return environmentReservationService.createReservation(environmentReservation);
    }

    @Timed
    @Override
    public EnvironmentReservation getById(String environmentReservationId) {
        environmentReservationSecurity.checkView();
        return environmentReservationService.findReservationById(environmentReservationId);
    }

    @Timed
    @Override
    public EnvironmentReservation update(EnvironmentReservation environmentReservation) {
        environmentReservationSecurity.checkEdit();
        return environmentReservationService.updateReservation(environmentReservation);
    }

    @Timed
    @Override
    public Map<String, List<EnvironmentReservation>> search(ReservationFilters filters) {
        environmentReservationSecurity.checkView();
        return environmentReservationService.searchReservations(filters)
                .entrySet()
                .stream()
                .collect(
                        Collectors.toMap(
                                entry -> entry.getKey().getId(),
                                Map.Entry::getValue
                        )
                );
    }

    @Timed
    @Override
    public boolean hasReservation(String environmentId, String applicationId) {
        environmentReservationSecurity.checkView();
        return environmentReservationService.existsReservation(environmentId, applicationId);
    }

    @Timed
    @Override
    public Date nearestComingReservation(String environmentId, String applicationId) {
        environmentReservationSecurity.checkView();
        return environmentReservationService.nearestComingReservation(environmentId, applicationId);
    }
}
