package com.xebialabs.xlrelease.environments.repository.sql.persistence

import com.xebialabs.deployit.exception.NotFoundException
import com.xebialabs.xlrelease.db.sql.SqlBuilder.Dialect
import com.xebialabs.xlrelease.db.sql.transaction.IsTransactional
import com.xebialabs.xlrelease.db.sql.{LimitOffset, SqlWithParameters}
import com.xebialabs.xlrelease.domain.environments.{Environment, EnvironmentReservation}
import com.xebialabs.xlrelease.environments.repository.sql.persistence.builder.EnvironmentsWithReservationsSqlBuilder
import com.xebialabs.xlrelease.environments.repository.sql.persistence.schema.ApplicationSchema.APPLICATIONS
import com.xebialabs.xlrelease.environments.repository.sql.persistence.schema.EnvironmentReservationSchema
import com.xebialabs.xlrelease.environments.repository.sql.persistence.schema.EnvironmentReservationSchema.{ENV_RES_TO_APP, ENV_RESERVATIONS => ER}
import com.xebialabs.xlrelease.environments.repository.sql.persistence.schema.EnvironmentSchema.ENVIRONMENTS
import com.xebialabs.xlrelease.repository.sql.persistence.CiId.CiId
import com.xebialabs.xlrelease.repository.sql.persistence.Utils.{params, _}
import com.xebialabs.xlrelease.repository.sql.persistence.{CiUid, PersistenceSupport}
import com.xebialabs.xlrelease.service.CiIdService
import org.springframework.beans.factory.annotation.{Autowired, Qualifier}
import org.springframework.jdbc.core.JdbcTemplate
import org.springframework.stereotype.Repository
import org.springframework.util.CollectionUtils

import java.util.Date
import scala.jdk.CollectionConverters._
import scala.util.Try

@Repository
@IsTransactional
class EnvironmentReservationPersistence @Autowired()(@Qualifier("xlrRepositoryJdbcTemplate") implicit val jdbcTemplate: JdbcTemplate,
                                                     @Qualifier("xlrRepositorySqlDialect") implicit val dialect: Dialect,
                                                     implicit val ciIdService: CiIdService,
                                                     val environmentPersistence: EnvironmentPersistence,
                                                     val applicationPersistence: ApplicationPersistence)
  extends PersistenceSupport with LimitOffset {

  def insert(reservation: EnvironmentReservation): CiId = {
    sanitizeReservationInput(reservation)

    val environmentUid = environmentPersistence.findUidById(reservation.getEnvironment.getId)
      .getOrElse(throw new NotFoundException(s"Environment [${reservation.getEnvironment.getId}] not found"))

    val applicationIdsToUids = getApplicationUids(reservation)

    val insertReservationStmt =
      s"""INSERT INTO ${ER.TABLE} (
         |${ER.ID},
         |${ER.ENVIRONMENT_UID},
         |${ER.START_DATE},
         |${ER.END_DATE},
         |${ER.NOTE})
         |VALUES (
         |:${ER.ID},
         |:${ER.ENVIRONMENT_UID},
         |:${ER.START_DATE},
         |:${ER.END_DATE},
         |:${ER.NOTE})
       """.stripMargin
    val resId = createPersistedId[EnvironmentReservation]
    sqlInsert(pkName(ER.CI_UID),
      insertReservationStmt,
      params(ER.ID -> resId,
        ER.ENVIRONMENT_UID -> environmentUid,
        ER.START_DATE -> reservation.getStartDate,
        ER.END_DATE -> reservation.getEndDate,
        ER.NOTE -> reservation.getNote),
      (ciUid: CiUid) => insertEnvResToAppReferences(ciUid, applicationIdsToUids.valuesIterator.toSet)
    )

    toDisplayId(resId)
  }

  def findById(environmentReservationId: CiId): Option[EnvironmentReservation] = {
    val (sql, params) = EnvironmentsWithReservationsSqlBuilder().withReservationId(toPersistedId(environmentReservationId)).build()
    jdbcTemplate.query(sql, params.toArray, Mappers.reservationResultSetExtractor)
  }

  def findUidById(reservationId: CiId): Option[CiUid] = {
    val stmt =
      s"""|SELECT ${ER.CI_UID}
          |FROM ${ER.TABLE}
          |WHERE ${ER.ID} = :${ER.ID}""".stripMargin
    sqlQuery(stmt, params(ER.ID -> toPersistedId(reservationId)), rs => CiUid(rs.getInt(ER.CI_UID))).headOption
  }

  def update(reservation: EnvironmentReservation): Boolean = {
    val reservationUid = findUidById(reservation.getId)
      .getOrElse(throw new NotFoundException(s"Reservation [${reservation.getId}] not found"))
    val environmentUid = environmentPersistence.findUidById(reservation.getEnvironment.getId)
      .getOrElse(throw new NotFoundException(s"Environment [${reservation.getEnvironment.getId}] not found"))

    sanitizeReservationInput(reservation)

    updateEnvResToAppReferences(reservation, reservationUid)

    val updateReservationStmt =
      s"""
         |UPDATE ${ER.TABLE}
         | SET
         |  ${ER.START_DATE} = :${ER.START_DATE},
         |  ${ER.END_DATE} = :${ER.END_DATE},
         |  ${ER.NOTE} = :${ER.NOTE},
         |  ${ER.ENVIRONMENT_UID} = :${ER.ENVIRONMENT_UID}
         | WHERE
         |  ${ER.CI_UID} = :${ER.CI_UID}
         |""".stripMargin
    sqlUpdate(updateReservationStmt,
      params(
        ER.START_DATE -> reservation.getStartDate,
        ER.END_DATE -> reservation.getEndDate,
        ER.NOTE -> reservation.getNote,
        ER.ENVIRONMENT_UID -> environmentUid,
        ER.CI_UID -> reservationUid
      ),
      _ == 1
    )
  }

  def search(sqlWithParameters: SqlWithParameters): Map[Environment, Seq[EnvironmentReservation]] = {
    val (sql, params) = sqlWithParameters
    jdbcTemplate.query(
      sql,
      params.toArray,
      Mappers.reservationSearchResultSetExtractor
    )
  }

  def delete(environmentReservationId: CiId): Try[Boolean] = {
    val stmt =
      s"""
         |DELETE FROM ${ER.TABLE}
         | WHERE ${ER.ID} = :reservationId
       """.stripMargin
    sqlExec(stmt, params("reservationId" -> toPersistedId(environmentReservationId)), ps => Try(ps.execute()))
  }

  def exists(environmentId: String, applicationId: String, date: Date): Boolean = {
    val stmt =
      s"""
         |SELECT 1
         |FROM ${ER.TABLE}
         |WHERE ${ER.START_DATE} <= :dateToSearch
         | AND ${ER.END_DATE} >= :dateToSearch
         | AND ${ER.ENVIRONMENT_UID} = (
         |  SELECT ${ENVIRONMENTS.CI_UID}
         |  FROM ${ENVIRONMENTS.TABLE}
         |  WHERE ${ENVIRONMENTS.ID} = :environmentId
         | )
         | AND ${ER.CI_UID} IN (
         |  SELECT ${ENV_RES_TO_APP.RESERVATION_UID}
         |  FROM ${ENV_RES_TO_APP.TABLE}
         |  WHERE ${ENV_RES_TO_APP.APPLICATION_UID} = (
         |   SELECT ${APPLICATIONS.CI_UID}
         |   FROM ${APPLICATIONS.TABLE}
         |   WHERE ${APPLICATIONS.ID} = :applicationId
         |  )
         | )
     """.stripMargin
    findOne {
      sqlQuery(stmt,
        params("environmentId" -> toPersistedId(environmentId), "applicationId" -> toPersistedId(applicationId), "dateToSearch" -> date),
        _.getInt(1) == 1)
    }.getOrElse(false)
  }

  def findNearestComing(environmentId: String, applicationId: String, date: Date): Option[Date] = {
    val stmt =
      s"""
         |SELECT ${ER.START_DATE}
         |FROM ${ER.TABLE}
         |WHERE ${ER.END_DATE} >= :dateToSearch
         | AND ${ER.ENVIRONMENT_UID} = (
         |  SELECT ${ENVIRONMENTS.CI_UID}
         |  FROM ${ENVIRONMENTS.TABLE}
         |  WHERE ${ENVIRONMENTS.ID} = :environmentId
         | )
         | AND ${ER.CI_UID} IN (
         |  SELECT ${ENV_RES_TO_APP.RESERVATION_UID}
         |  FROM ${ENV_RES_TO_APP.TABLE}
         |  WHERE ${ENV_RES_TO_APP.APPLICATION_UID} = (
         |   SELECT ${APPLICATIONS.CI_UID}
         |   FROM ${APPLICATIONS.TABLE}
         |   WHERE ${APPLICATIONS.ID} = :applicationId
         |  )
         | )
         |ORDER BY ${ER.START_DATE} ASC
     """.stripMargin
    findOne {
      sqlQuery(addLimitAndOffset(stmt, Option(1)),
        params("environmentId" -> toPersistedId(environmentId), "applicationId" -> toPersistedId(applicationId), "dateToSearch" -> date),
        r => Option(r.getTimestamp(1)))
    }.getOrElse(None)
  }

  private def getApplicationUids(reservation: EnvironmentReservation): Map[CiId, CiUid] = {
    if (CollectionUtils.isEmpty(reservation.getApplications)) {
      Map.empty[CiId, CiUid]
    } else {
      val applicationIds = reservation.getApplications.asScala.map(_.getId).toSet
      val applicationIdsToUids = applicationPersistence.getUidsByIds(applicationIds)
      val applicationDiff = applicationIds.diff(applicationIdsToUids.keySet)
      if (applicationDiff.nonEmpty) {
        throw new NotFoundException(s"Applications [${applicationDiff.mkString(", ")}] not found")
      }
      applicationIdsToUids
    }
  }

  private def insertEnvResToAppReferences(reservationUid: CiUid, appUids: Set[CiUid]) = {
    val insertEnvResToAppStmt =
      s"""INSERT INTO ${ENV_RES_TO_APP.TABLE} (
         |${ENV_RES_TO_APP.RESERVATION_UID},
         |${ENV_RES_TO_APP.APPLICATION_UID}
         |)
         |VALUES (
         |:${ENV_RES_TO_APP.RESERVATION_UID},
         |:${ENV_RES_TO_APP.APPLICATION_UID}
         |)
         |""".stripMargin

    sqlBatch(insertEnvResToAppStmt,
      appUids.map { appUid =>
        params(
          ENV_RES_TO_APP.RESERVATION_UID -> reservationUid,
          ENV_RES_TO_APP.APPLICATION_UID -> appUid
        )
      }
    )
  }

  private def updateEnvResToAppReferences(reservation: EnvironmentReservation, reservationUid: CiUid) = {
    val applicationIdsToUids = getApplicationUids(reservation)

    val deleteEnvResToAppRefsStmt =
      s"""
         |DELETE FROM ${ENV_RES_TO_APP.TABLE}
         | WHERE
         | ${ENV_RES_TO_APP.RESERVATION_UID} = :${ENV_RES_TO_APP.RESERVATION_UID}
       """.stripMargin
    sqlExec(deleteEnvResToAppRefsStmt, params(ENV_RES_TO_APP.RESERVATION_UID -> reservationUid), _.execute())
    insertEnvResToAppReferences(reservationUid, applicationIdsToUids.valuesIterator.toSet)
  }

  private def sanitizeReservationInput(reservation: EnvironmentReservation): Unit = {
    reservation.setNote(reservation.getNote.trimAndTruncate(EnvironmentReservationSchema.NOTE_LENGTH))
  }
}
