package com.xebialabs.xlrelease.environments.repository.sql.persistence

import com.xebialabs.deployit.exception.NotFoundException
import com.xebialabs.xlrelease.db.sql.SqlBuilder.Dialect
import com.xebialabs.xlrelease.db.sql.SqlWithParameters
import com.xebialabs.xlrelease.db.sql.transaction.IsTransactional
import com.xebialabs.xlrelease.domain.environments.EnvironmentStage
import com.xebialabs.xlrelease.environments.repository.sql.persistence.builder.ColumnAliases
import com.xebialabs.xlrelease.environments.repository.sql.persistence.schema.EnvironmentSchema.ENVIRONMENTS
import com.xebialabs.xlrelease.environments.repository.sql.persistence.schema.EnvironmentStageSchema
import com.xebialabs.xlrelease.environments.repository.sql.persistence.schema.EnvironmentStageSchema.ENV_STAGES
import com.xebialabs.xlrelease.repository.sql.persistence.CiId.CiId
import com.xebialabs.xlrelease.repository.sql.persistence.Utils._
import com.xebialabs.xlrelease.repository.sql.persistence.{CiUid, PersistenceSupport}
import com.xebialabs.xlrelease.service.CiIdService
import org.springframework.beans.factory.annotation.{Autowired, Qualifier}
import org.springframework.jdbc.core.JdbcTemplate
import org.springframework.stereotype.Repository

import scala.jdk.CollectionConverters._

@Repository
@IsTransactional
class EnvironmentStagePersistence @Autowired()(@Qualifier("xlrRepositoryJdbcTemplate") implicit val jdbcTemplate: JdbcTemplate,
                                               @Qualifier("xlrRepositorySqlDialect") implicit val dialect: Dialect,
                                               implicit val ciIdService: CiIdService)
  extends PersistenceSupport {

  private val STMT_EXISTS_BY_TITLE_IGNORECASE =
    s"""
       |SELECT COUNT(*)
       |FROM ${ENV_STAGES.TABLE}
       |WHERE LOWER(${ENV_STAGES.TITLE}) = LOWER(:${ENV_STAGES.TITLE})
      """.stripMargin

  def insert(environmentStage: EnvironmentStage): EnvironmentStage = {
    sanitizeStageInput(environmentStage)

    val exists = sqlQuery(STMT_EXISTS_BY_TITLE_IGNORECASE, params(ENV_STAGES.TITLE -> environmentStage.getTitle), _.getInt(1) > 0).head
    if (exists) {
      throw new IllegalArgumentException(s"Environment stage with title '${environmentStage.getTitle}' already exists")
    }

    val stmt =
      s"""
         |INSERT INTO ${ENV_STAGES.TABLE} (
         |${ENV_STAGES.ID},
         |${ENV_STAGES.TITLE})
         |VALUES (
         |:${ENV_STAGES.ID},
         |:${ENV_STAGES.TITLE})
        """.stripMargin
    val stageId = createPersistedId[EnvironmentStage]
    sqlInsert(pkName(ENV_STAGES.CI_UID),
      stmt,
      params(ENV_STAGES.ID -> stageId,
        ENV_STAGES.TITLE -> environmentStage.getTitle)
    )
    val stage = new EnvironmentStage
    stage.setId(toDisplayId(stageId))
    stage.setTitle(environmentStage.getTitle)
    stage
  }

  private val STMT_EXISTS_ANOTHER_ENV_STAGE_WITH_TITLE =
    s"""
       |SELECT COUNT(*)
       |FROM ${ENV_STAGES.TABLE}
       |WHERE LOWER(${ENV_STAGES.TITLE}) = LOWER(:${ENV_STAGES.TITLE})
       |AND ${ENV_STAGES.ID} <> :${ENV_STAGES.ID}
      """.stripMargin

  def update(environmentStage: EnvironmentStage): Boolean = {
    sanitizeStageInput(environmentStage)

    val exists = sqlQuery(STMT_EXISTS_ANOTHER_ENV_STAGE_WITH_TITLE,
      params(ENV_STAGES.TITLE -> environmentStage.getTitle,
        ENV_STAGES.ID -> toPersistedId(environmentStage.getId)),
      _.getInt(1) > 0).head

    if (exists) {
      throw new IllegalArgumentException(s"Environment stage with title '${environmentStage.getTitle}' already exists")
    }

    val stmt =
      s"""|UPDATE ${ENV_STAGES.TABLE}
          |SET ${ENV_STAGES.TITLE} = :${ENV_STAGES.TITLE}
          |WHERE ${ENV_STAGES.ID} = :${ENV_STAGES.ID}
      """.stripMargin
    sqlUpdate(
      stmt,
      params(ENV_STAGES.TITLE -> environmentStage.getTitle,
        ENV_STAGES.ID -> toPersistedId(environmentStage.getId)),
      _ == 1
    )
  }

  def findById(environmentStageId: CiId): Option[EnvironmentStage] = {
    val stmt =
      s"""|SELECT ${ENV_STAGES.ID} ${ColumnAliases.EnvStages.ID},
          |${ENV_STAGES.TITLE} ${ColumnAliases.EnvStages.TITLE}
          |FROM ${ENV_STAGES.TABLE}
          |WHERE ${ENV_STAGES.ID} = :${ENV_STAGES.ID}
       """.stripMargin
    sqlQuery(stmt,
      params(ENV_STAGES.ID -> toPersistedId(environmentStageId)),
      Mappers.environmentStageMapper
    ).headOption
  }

  def findByTitle(environmentStageTitle: String): Option[EnvironmentStage] = {
    val stmt =
      s"""SELECT ${ENV_STAGES.ID} ${ColumnAliases.EnvStages.ID},
         |${ENV_STAGES.TITLE} ${ColumnAliases.EnvStages.TITLE}
         |FROM ${ENV_STAGES.TABLE}
         |WHERE LOWER(${ENV_STAGES.TITLE}) = LOWER(:${ENV_STAGES.TITLE})
       """.stripMargin
    sqlQuery(stmt, params(ENV_STAGES.TITLE -> environmentStageTitle), Mappers.environmentStageMapper).headOption
  }

  def search(sqlWithParameters: SqlWithParameters): Seq[EnvironmentStage] = {
    val (sql, params) = sqlWithParameters
    jdbcTemplate.query[EnvironmentStage](sql, params.toArray, Mappers.environmentStageMapper).asScala.toSeq
  }

  def delete(environmentStageId: CiId): Boolean = {
    val stmt =
      s"""|DELETE FROM ${ENV_STAGES.TABLE}
          |WHERE ${ENV_STAGES.ID} = :${ENV_STAGES.ID}
      """.stripMargin
    sqlUpdate(stmt, params(ENV_STAGES.ID -> toPersistedId(environmentStageId)), _ == 1)
  }

  def usedBy(environmentStageId: CiId): Seq[CiId] = {
    val stmt =
      s"""
         |SELECT ${ENVIRONMENTS.ID} FROM ${ENVIRONMENTS.TABLE}
         |WHERE ${ENVIRONMENTS.ENV_STAGE_UID} = :${ENV_STAGES.CI_UID}
       """.stripMargin
    val stageUid = findUidById(environmentStageId)
      .getOrElse(throw new NotFoundException(s"Environment stage [$environmentStageId] not found"))

    sqlQuery(stmt, params(ENV_STAGES.CI_UID -> stageUid), (rs, _) => rs.getCiId(ENVIRONMENTS.ID)).toSeq
  }

  def findUidById(environmentStageId: CiId): Option[CiUid] = {
    val stmt =
      s"""|SELECT ${ENV_STAGES.CI_UID}
          |FROM ${ENV_STAGES.TABLE}
          |WHERE ${ENV_STAGES.ID} = :${ENV_STAGES.ID}""".stripMargin
    sqlQuery(stmt, params(ENV_STAGES.ID -> toPersistedId(environmentStageId)),
      rs => CiUid(rs.getInt(ENV_STAGES.CI_UID))
    ).headOption
  }

  private def sanitizeStageInput(stage: EnvironmentStage): Unit = {
    stage.setTitle(stage.getTitle.trimAndTruncate(EnvironmentStageSchema.TITLE_LENGTH))
  }
}
