package com.xebialabs.xlrelease.api.v1.impl;

import java.util.List;
import java.util.stream.Collectors;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Controller;
import com.codahale.metrics.annotation.Timed;

import com.xebialabs.xlrelease.api.v1.EnvironmentLabelApi;
import com.xebialabs.xlrelease.api.v1.filter.EnvironmentLabelFilters;
import com.xebialabs.xlrelease.api.v1.form.EnvironmentLabelForm;
import com.xebialabs.xlrelease.api.v1.views.EnvironmentLabelView;
import com.xebialabs.xlrelease.domain.environments.EnvironmentLabel;
import com.xebialabs.xlrelease.environments.service.EnvironmentLabelService;
import com.xebialabs.xlrelease.environments.service.EnvironmentSecurity;

import static com.xebialabs.xlrelease.api.v1.views.EnvironmentLabelView.fromEnvironmentLabel;

@Controller
public class EnvironmentLabelApiImpl implements EnvironmentLabelApi {

    private EnvironmentLabelService environmentLabelService;
    private EnvironmentSecurity environmentSecurity;

    @Autowired
    public EnvironmentLabelApiImpl(EnvironmentLabelService environmentLabelService, EnvironmentSecurity environmentSecurity) {
        this.environmentLabelService = environmentLabelService;
        this.environmentSecurity = environmentSecurity;
    }

    @Timed
    @Override
    public EnvironmentLabelView createLabel(EnvironmentLabelForm labelForm) {
        environmentSecurity.checkCreate();
        return fromEnvironmentLabel(
                environmentLabelService.createLabel(labelForm.toLabel())
        );
    }

    @Timed
    @Override
    public EnvironmentLabelView getLabelById(String environmentLabelId) {
        environmentSecurity.checkView();
        return fromEnvironmentLabel(this.environmentLabelService.findLabelById(environmentLabelId));
    }

    @Timed
    @Override
    public EnvironmentLabelView updateLabel(String environmentLabelId, EnvironmentLabelForm labelForm) {
        environmentSecurity.checkEdit();
        EnvironmentLabel label = labelForm.toLabel();
        label.setId(environmentLabelId);
        return fromEnvironmentLabel(environmentLabelService.updateLabel(label));
    }

    @Timed
    @Override
    public List<EnvironmentLabelView> searchLabels(EnvironmentLabelFilters filters) {
        environmentSecurity.checkView();
        return environmentLabelService.searchLabels(filters)
                .stream().map(EnvironmentLabelView::fromEnvironmentLabel).collect(Collectors.toList());
    }

    @Timed
    @Override
    public void delete(String environmentLabelId) {
        environmentSecurity.checkEdit();
        environmentLabelService.deleteLabel(environmentLabelId);
    }

    @Timed
    @Override
    public EnvironmentLabel create(EnvironmentLabel environmentLabel) {
        environmentSecurity.checkCreate();
        return environmentLabelService.createLabel(environmentLabel);
    }

    @Timed
    @Override
    public EnvironmentLabel getById(String id) {
        environmentSecurity.checkView();
        return this.environmentLabelService.findLabelById(id);
    }

    @Timed
    @Override
    public EnvironmentLabel update(EnvironmentLabel environmentLabel) {
        environmentSecurity.checkEdit();
        return environmentLabelService.updateLabel(environmentLabel);
    }

    @Timed
    @Override
    public List<EnvironmentLabel> search(EnvironmentLabelFilters filters) {
        environmentSecurity.checkView();
        return environmentLabelService.searchLabels(filters);
    }
}
