package com.xebialabs.xlrelease.api.internal

import com.xebialabs.deployit.security.permission.PlatformPermissions.ADMIN
import com.xebialabs.xlrelease.api.v1.filter.EnvironmentStageFilters
import com.xebialabs.xlrelease.domain.environments.{Application, Environment, EnvironmentReservation, EnvironmentStage}
import com.xebialabs.xlrelease.environments.repository.{ApplicationRepository, EnvironmentRepository, EnvironmentReservationRepository, EnvironmentStageRepository}
import com.xebialabs.xlrelease.repository.Page
import com.xebialabs.xlrelease.security.PermissionChecker
import grizzled.slf4j.Logging
import org.jboss.resteasy.annotations.cache.NoCache
import org.springframework.beans.factory.annotation.Autowired
import org.springframework.stereotype.Controller

import java.lang.String.format
import javax.ws.rs._
import javax.ws.rs.core.MediaType


@Path("/fixtures")
@Consumes(Array(MediaType.APPLICATION_JSON))
@Produces(Array(MediaType.APPLICATION_JSON))
@Controller
@NoCache
class FixtureEnvironmentManagementResource @Autowired()(permissionChecker: PermissionChecker,
                                                        applicationRepository: ApplicationRepository,
                                                        environmentRepository: EnvironmentRepository,
                                                        environmentStageRepository: EnvironmentStageRepository,
                                                        environmentReservationRepository: EnvironmentReservationRepository)
  extends Logging {

  @Path("application")
  @POST def createApplication(application: Application): Application = {
    permissionChecker.check(ADMIN)
    logger.debug(format("Creating application: %s", application))
    applicationRepository.createApplication(application)
  }

  @Path("environment")
  @POST def createEnvironment(environment: Environment): Environment = {
    permissionChecker.check(ADMIN)
    logger.debug(format("Creating environment: %s", environment))

    val environmentStage = if (environment.getStage.getId == null) {
      val EnvironmentStageFilters = new EnvironmentStageFilters
      EnvironmentStageFilters.setTitle(environment.getStage.getTitle)

      environmentStageRepository.search(EnvironmentStageFilters, Page(0, Int.MaxValue, 0)).headOption.getOrElse {
        val newEnvironmentStage = new EnvironmentStage
        newEnvironmentStage.setTitle(environment.getStage.getTitle)
        environmentStageRepository.create(newEnvironmentStage)
      }
    } else {
      environmentStageRepository.findById(environment.getStage.getId)
    }
    environment.setStage(environmentStage)
    environmentRepository.createEnvironment(environment)
  }

  @Path("reservation")
  @POST def createReservation(environmentReservation: EnvironmentReservation): EnvironmentReservation = {
    permissionChecker.check(ADMIN)
    environmentReservationRepository.create(environmentReservation)
  }

  @Path("/{id:.*/EnvironmentReservation[^/]*}")
  @DELETE def deleteReservation(@PathParam("id") id: String): Unit = {
    permissionChecker.check(ADMIN)
    logger.debug(format("Deleting environment reservation entity: [%s]", id))
    environmentReservationRepository.delete(id)
  }

  @Path("/{applicationId:.*/Application[^/]*}")
  @DELETE def deleteApplication(@PathParam("applicationId") applicationId: String): Unit = {
    permissionChecker.check(ADMIN)
    logger.debug(format("Deleting application entity: [%s]", applicationId))
    applicationRepository.deleteApplication(applicationId)
  }

  @Path("/{environmentId:.*/Environment[^/]*}")
  @DELETE def deleteEnvironment(@PathParam("environmentId") environmentId: String): Unit = {
    permissionChecker.check(ADMIN)
    logger.debug(format("Deleting environment entity: [%s]", environmentId))
    environmentRepository.deleteEnvironment(environmentId)
  }
}
