package com.xebialabs.xlrelease.environments.repository.sql.persistence

import com.xebialabs.xlrelease.db.sql.SqlBuilder.Dialect
import com.xebialabs.xlrelease.db.sql.SqlWithParameters
import com.xebialabs.xlrelease.db.sql.transaction.IsTransactional
import com.xebialabs.xlrelease.domain.environments.EnvironmentLabel
import com.xebialabs.xlrelease.environments.repository.sql.persistence.builder.ColumnAliases
import com.xebialabs.xlrelease.environments.repository.sql.persistence.schema.EnvironmentLabelSchema
import com.xebialabs.xlrelease.environments.repository.sql.persistence.schema.EnvironmentLabelSchema.ENV_LABELS
import com.xebialabs.xlrelease.repository.sql.persistence.CiId.CiId
import com.xebialabs.xlrelease.repository.sql.persistence.Utils._
import com.xebialabs.xlrelease.repository.sql.persistence.{CiUid, PersistenceSupport}
import com.xebialabs.xlrelease.service.CiIdService
import org.springframework.beans.factory.annotation.{Autowired, Qualifier}
import org.springframework.jdbc.core.JdbcTemplate
import org.springframework.stereotype.Repository

import scala.jdk.CollectionConverters._
import scala.util.Try

@Repository
@IsTransactional
class EnvironmentLabelPersistence @Autowired()(@Qualifier("xlrRepositoryJdbcTemplate") implicit val jdbcTemplate: JdbcTemplate,
                                               @Qualifier("xlrRepositorySqlDialect") implicit val dialect: Dialect,
                                               implicit val ciIdService: CiIdService)
  extends PersistenceSupport {

  private val STMT_EXISTS_BY_TITLE_IGNORECASE =
    s"""
       |SELECT COUNT(*)
       |FROM ${ENV_LABELS.TABLE}
       |WHERE LOWER(${ENV_LABELS.TITLE}) = LOWER(:${ENV_LABELS.TITLE})
      """.stripMargin

  def insert(environmentLabel: EnvironmentLabel): EnvironmentLabel = {
    val exists = sqlQuery(STMT_EXISTS_BY_TITLE_IGNORECASE, params(ENV_LABELS.TITLE -> environmentLabel.getTitle), _.getInt(1) > 0).head
    if (exists) {
      throw new IllegalArgumentException(s"Environment label with title '${environmentLabel.getTitle}' already exists")
    }

    val stmt =
      s"""
         |INSERT INTO ${ENV_LABELS.TABLE} (
         |${ENV_LABELS.ID},
         |${ENV_LABELS.TITLE},
         |${ENV_LABELS.COLOR})
         |VALUES (
         |:${ENV_LABELS.ID},
         |:${ENV_LABELS.TITLE},
         |:${ENV_LABELS.COLOR})
        """.stripMargin
    val truncatedTitle = environmentLabel.getTitle.truncate(EnvironmentLabelSchema.TITLE_COLUMN_LENGTH)
    val labelId = createPersistedId[EnvironmentLabel]
    sqlInsert(pkName(ENV_LABELS.CI_UID),
      stmt,
      params(ENV_LABELS.ID -> labelId,
        ENV_LABELS.TITLE -> truncatedTitle,
        ENV_LABELS.COLOR -> environmentLabel.getColor)
    )
    val label = new EnvironmentLabel
    label.setId(toDisplayId(labelId))
    label.setTitle(truncatedTitle)
    label.setColor(environmentLabel.getColor)
    label
  }

  private val STMT_EXISTS_ANOTHER_ENV_LABEL_WITH_TITLE =
    s"""
       |SELECT COUNT(*)
       |FROM ${ENV_LABELS.TABLE}
       |WHERE LOWER(${ENV_LABELS.TITLE}) = LOWER(:${ENV_LABELS.TITLE})
       |AND ${ENV_LABELS.ID} <> :${ENV_LABELS.ID}
      """.stripMargin

  def update(environmentLabel: EnvironmentLabel): Boolean = {
    val exists = sqlQuery(STMT_EXISTS_ANOTHER_ENV_LABEL_WITH_TITLE,
      params(ENV_LABELS.TITLE -> environmentLabel.getTitle,
        ENV_LABELS.ID -> toPersistedId(environmentLabel.getId)),
      _.getInt(1) > 0).head

    if (exists) {
      throw new IllegalArgumentException(s"Environment label with title '${environmentLabel.getTitle}' already exists")
    }

    val stmt =
      s"""|UPDATE ${ENV_LABELS.TABLE}
          | SET
          |  ${ENV_LABELS.TITLE} = :labelTitle,
          |  ${ENV_LABELS.COLOR} = :labelColor
          |WHERE ${ENV_LABELS.ID} = :labelId
      """.stripMargin
    sqlUpdate(
      stmt,
      params(
        "labelTitle" -> environmentLabel.getTitle.truncate(EnvironmentLabelSchema.TITLE_COLUMN_LENGTH),
        "labelColor" -> environmentLabel.getColor,
        "labelId" -> toPersistedId(environmentLabel.getId)),
      _ == 1
    )
  }

  def findById(environmentLabelId: CiId): Option[EnvironmentLabel] = {
    val stmt =
      s"""|SELECT ${ENV_LABELS.ID} ${ColumnAliases.EnvLabels.ID},
          |${ENV_LABELS.TITLE} ${ColumnAliases.EnvLabels.TITLE},
          |${ENV_LABELS.COLOR} ${ColumnAliases.EnvLabels.COLOR}
          |FROM ${ENV_LABELS.TABLE}
          |WHERE ${ENV_LABELS.ID} = :labelId
       """.stripMargin
    sqlQuery(stmt, params("labelId" -> toPersistedId(environmentLabelId)), Mappers.environmentLabelMapper).headOption
  }

  def findByTitle(environmentLabelTitle: String): Option[EnvironmentLabel] = {
    val stmt =
      s"""|SELECT ${ENV_LABELS.ID} ${ColumnAliases.EnvLabels.ID},
          |${ENV_LABELS.TITLE} ${ColumnAliases.EnvLabels.TITLE},
          |${ENV_LABELS.COLOR} ${ColumnAliases.EnvLabels.COLOR}
          |FROM ${ENV_LABELS.TABLE}
          |WHERE LOWER(${ENV_LABELS.TITLE}) = LOWER(:${ENV_LABELS.TITLE})
       """.stripMargin
    sqlQuery(stmt, params(ENV_LABELS.TITLE -> environmentLabelTitle), Mappers.environmentLabelMapper).headOption
  }

  def delete(environmentLabelId: CiId): Try[Boolean] = {
    val stmt =
      s"""
         |DELETE FROM ${ENV_LABELS.TABLE}
         | WHERE ${ENV_LABELS.ID} = :environmentLabelId
       """.stripMargin
    sqlExec(stmt, params("environmentLabelId" -> toPersistedId(environmentLabelId)), ps => Try(ps.execute()))
  }

  def search(sqlWithParameters: SqlWithParameters): Seq[EnvironmentLabel] = {
    val (sql, params) = sqlWithParameters
    jdbcTemplate.query[EnvironmentLabel](sql, params.toArray, Mappers.environmentLabelMapper).asScala.toSeq
  }

  def getUidsByIds(labelIds: Iterable[CiId]): Map[CiId, CiUid] = {
    val stmt =
      s"""|SELECT ${ENV_LABELS.CI_UID}, ${ENV_LABELS.ID}
          |FROM ${ENV_LABELS.TABLE}
          |WHERE ${ENV_LABELS.ID} IN (:labelIds)
       """.stripMargin
    sqlQuery(stmt,
      params("labelIds" -> labelIds.map(toPersistedId).asJava),
      rs => rs.getCiId(ENV_LABELS.ID) -> CiUid(rs.getInt(ENV_LABELS.CI_UID))
    ).toMap
  }
}
