package com.xebialabs.xlrelease.plugins.dashboard.repository

import com.xebialabs.xlrelease.db.sql.transaction.IsTransactional
import com.xebialabs.xlrelease.limits.LimitEnforcer
import com.xebialabs.xlrelease.limits.LimitEnforcer.LimitType
import com.xebialabs.xlrelease.plugins.dashboard.domain.Dashboard
import com.xebialabs.xlrelease.repository.PersistenceInterceptor
import com.xebialabs.xlrelease.utils.TenantContext
import grizzled.slf4j.Logging
import org.springframework.stereotype.Component
import org.springframework.util.StringUtils.hasText

/**
 * Dashboard Limit Enforcement Rules:
 *
 * Dashboard Count Limits:
 * - Applied to: Home, Folder, Global dashboards (custom/user-created)
 * - Excluded: Release dashboards (built-in/auto-generated)
 * - Scope: Per-tenant
 *
 * Tile Count Limits:
 * - Applied to: ALL dashboard types (including Release dashboards)
 * - Scope: Per-dashboard
 * - Enforced: During create and update operations
 */
@Component
@IsTransactional
class DashboardLimitPersistenceInterceptor(limitEnforcer: LimitEnforcer,
                                           dashboardRepository: SqlDashboardRepository,
                                           releaseDashboardRepository: ReleaseDashboardRepository)
  extends PersistenceInterceptor[Dashboard] with Logging {

  dashboardRepository.registerPersistenceInterceptor(this)
  releaseDashboardRepository.registerPersistenceInterceptor(this)

  override def onCreate(dashboard: Dashboard): Unit = {
    val tenantId = ensureTenantId(dashboard)

    // Enforce dashboard count limit for all dashboards except built-in Release dashboards
    if (!dashboard.isReleaseDashboard) {
      limitEnforcer.enforceLimit(tenantId, LimitType.DASHBOARDS, 1, () => dashboardRepository.tenantDashboardCount(tenantId))
    }

    // Enforce tile count limit for all dashboards
    val tileCount = dashboard.getTiles.size()
    limitEnforcer.enforceLimit(tenantId, LimitType.DASHBOARD_TILES, 0, () => tileCount)
  }

  override def onUpdate(dashboard: Dashboard): Unit = {
    val tenantId = ensureTenantId(dashboard)
    val tileCount = dashboard.getTiles.size()
    limitEnforcer.enforceLimit(tenantId, LimitType.DASHBOARD_TILES, 0, () => tileCount)
  }

  private def ensureTenantId(ci: Dashboard): String = {
    if (hasText(ci.getTenantId)) {
      ci.getTenantId
    } else {
      val tenantIdFromContext = TenantContext.getTenant()
      ci.setTenantId(tenantIdFromContext)
      tenantIdFromContext
    }
  }
}
