import { call, put } from 'redux-saga/effects';
import { navigation } from '@xlr-ui/app/features/main-navigation/ducks/navigation.reducer';
import { registerSidebarProvider, SidebarProvider } from '@xlr-ui/app/features/custom-ui/sidebar.registry';
import getAngularService from '@xlr-ui/app/features/common/services/angular-accessor';
import { BreadcrumbItem, XlrDashboardService } from '@xlr-ui/app/features/tasks/types/angular';
import { Dashboard } from './types';
import { IHttpResponse } from 'angular';
import { DASHBOARD_LIST_SUFFIX } from './xlr-dashboard-constants';
import { MenuItem } from '@xlr-ui/app/types/synthetic';

const { setBackItem, updateBreadcrumbs, setSidebarMenu, setLastBreadcrumb } = navigation.actions;

export class DashboardSidebarProvider implements SidebarProvider {
    dashboardsPrefix = '/dashboards';
    dashboardDetailsPattern = /\/Tile(\w+)\/details$/;

    checkIfInDashboardDetails(path: string) {
        return this.dashboardDetailsPattern.test(path);
    }

    getDashboardId(path: string, isInDashboardDetails: boolean) {
        let dashboardId = path.replace('/dashboards/', '');
        if (isInDashboardDetails) {
            dashboardId = dashboardId.replace(this.dashboardDetailsPattern, '');
        }
        return dashboardId;
    }

    isSupported(path: string): boolean {
        return path.startsWith('/dashboards/Applications/Dashboard');
    }

    *provide(path: string) {
        const isInDashboardDetails = this.checkIfInDashboardDetails(path);
        const dashboardId = this.getDashboardId(path, isInDashboardDetails);
        const xlrDashboardService: XlrDashboardService = yield call(getAngularService, 'xlrelease.XlrDashboardService');
        const { data: dashboard }: IHttpResponse<Dashboard> = yield call([xlrDashboardService, xlrDashboardService.loadConfiguration], dashboardId);
        const dashboardHref = `${this.dashboardsPrefix}/${dashboard.id}`;

        const breadcrumbs: BreadcrumbItem[] = [
            { text: 'Reports', href: this.dashboardsPrefix },
            { text: 'Custom dashboards', href: `/${DASHBOARD_LIST_SUFFIX}` },
            { text: dashboard.title, href: dashboardHref },
        ];

        let backItem: MenuItem;

        if (isInDashboardDetails) {
            breadcrumbs.push({ text: 'Details' });
            backItem = {
                pathSuffix: dashboardHref,
                title: dashboard.title,
                text: dashboard.title,
            };
        } else {
            backItem = {
                pathSuffix: DASHBOARD_LIST_SUFFIX,
                title: 'Custom dashboards',
                text: 'Custom dashboards',
            };
        }

        yield put(setLastBreadcrumb(undefined));
        yield put(updateBreadcrumbs(breadcrumbs));
        yield put(setSidebarMenu({ sidebarMenu: [], withFolder: false }));
        yield put(setBackItem(backItem));
    }
}

registerSidebarProvider(new DashboardSidebarProvider(), 0);
