import React, { MouseEvent, useState } from 'react';
import { Provider } from 'react-redux';
import isEmpty from 'lodash/isEmpty';
import { store } from '@xlr-ui/app/js/app-config';
import { ButtonProps, DotButton, DotIcon, DotIconButton, DotThemeProvider, DotTypography } from '@digital-ai/dot-components';
import { Dashboard } from '../../types';
import { DashboardActionContextMenu } from './dashboard-action-context-menu.component';
import { Release } from '@xlr-ui/app/types';
import { ReleaseStatus } from '@xlr-ui/app/features/release-overview/release-status-label/release-status.component';
import { useCloseSidebar } from '@xlr-ui/app/features/common/hooks';
import { useAppDispatch, useAppSelector } from '@xlr-ui/app/js/hooks';
import { getNavigationState } from '@xlr-ui/app/features/main-navigation/ducks/navigation.selectors';
import './dashboard-details-header.component.less';

export interface DashboardDetailsHeaderProps {
    canCreateDashboard: boolean;
    canDeleteDashboard: boolean;
    canEditDashboard: boolean;
    dashboard: Dashboard;
    editButtonProps: Partial<ButtonProps>;
    exportAsPdfButtonProps: Partial<ButtonProps>;
    isDesignMode: boolean;
    isFolderDashboard: boolean;
    isGlobalDashboard: boolean;
    onDelete: (dashboardId: string) => void;
    onDuplicate: (dashboard: Dashboard) => void;
    refreshButtonProps: Partial<ButtonProps>;
    release?: Release;
}

export const DashboardDetailsHeader = (props: DashboardDetailsHeaderProps) => {
    return (
        <Provider store={store}>
            <DotThemeProvider>
                <DashboardDetailsHeaderContent {...props} />
            </DotThemeProvider>
        </Provider>
    );
};

export const DashboardDetailsHeaderContent = ({
    canCreateDashboard,
    canDeleteDashboard,
    canEditDashboard,
    dashboard,
    editButtonProps,
    exportAsPdfButtonProps,
    isDesignMode,
    isFolderDashboard,
    isGlobalDashboard,
    onDelete,
    onDuplicate,
    refreshButtonProps,
    release,
}: DashboardDetailsHeaderProps) => {
    const dispatch = useAppDispatch();
    const { menuIsOpen } = useAppSelector(getNavigationState);
    const [contextMenuAnchorEl, setContextMenuAnchorEl] = useState<HTMLElement | null>(null);
    const isReleaseDashboard = !isEmpty(release);
    const isDeleteEnabled = isGlobalDashboard && canDeleteDashboard;
    const isContextMenuAvailable = isDeleteEnabled || canCreateDashboard;
    const isEditButtonAvailable = canEditDashboard && (isGlobalDashboard || isFolderDashboard);

    isGlobalDashboard && useCloseSidebar(menuIsOpen, dispatch);

    const handleContextMenuOpen = (event: MouseEvent<HTMLElement>) => {
        setContextMenuAnchorEl(event.currentTarget);
    };

    const headerTitle = isReleaseDashboard ? `Release dashboard for ${release.title}` : dashboard.title;

    return (
        <div className="dashboard-details-header">
            <div className="dashboard-details-header-left">
                <DotTypography data-testid="header-title" noWrap={true} variant="h1">
                    {headerTitle}
                </DotTypography>
                {isReleaseDashboard && <ReleaseStatus status={release.status} />}
            </div>

            <div className="dashboard-details-header-actions">
                {isDesignMode ? (
                    isEditButtonAvailable && (
                        <DotButton data-testid="edit-properties-btn" startIcon={<DotIcon iconId="edit" />} type="outlined" {...editButtonProps}>
                            Edit properties
                        </DotButton>
                    )
                ) : (
                    <>
                        <DotButton data-testid="refresh-dashboard-btn" startIcon={<DotIcon iconId="refresh" />} type="text" {...refreshButtonProps}>
                            Refresh dashboard
                        </DotButton>
                        <DotButton data-testid="export-as-pdf-btn" startIcon={<DotIcon iconId="download" />} type="outlined" {...exportAsPdfButtonProps}>
                            Export as PDF
                        </DotButton>
                    </>
                )}
                {isContextMenuAvailable && <DotIconButton iconId="options" onClick={handleContextMenuOpen} />}
            </div>
            {isContextMenuAvailable && (
                <DashboardActionContextMenu
                    anchorEl={contextMenuAnchorEl || undefined}
                    dashboard={dashboard}
                    hasCreateDashboardPermission={canCreateDashboard}
                    hasDeleteDashboardPermission={canDeleteDashboard}
                    onDelete={onDelete}
                    onDuplicate={onDuplicate}
                    onLeave={() => setContextMenuAnchorEl(null)}
                />
            )}
        </div>
    );
};
