import React from 'react';
import { DashboardCards, DashboardCardsProps } from './dashboard-cards.component';
import { dashboardsMock } from '../../../mocks';
import { mountWithTheme, ReactWrapper } from '@xlr-ui/tests/unit/testing-utils';
import { DotCard, DotEmptyState, DotIcon, DotIconButton, DotInputText, DotLink, DotMenu, DotTypography } from '@digital-ai/dot-components';
import { DashboardActionContextMenu } from '../dashboard-action-context-menu.component';
import { foldersMock } from '@xlr-ui/app/features/tasks/__mocks__/release.mock';
import * as angularAccessor from '@xlr-ui/app/features/common/services/angular-accessor';
import { DEFAULT_FOLDER_TAB } from '@xlr-ui/app/features/folders/folder-constants';

describe('DashboardCards', () => {
    let wrapper: ReactWrapper;
    const stopPropagation = jest.fn();
    const onDelete = jest.fn();
    const onDuplicate = jest.fn();
    const onDashboardFilterChange = jest.fn();
    const dashboards = dashboardsMock;

    const defaultProps: DashboardCardsProps = {
        dashboards,
        hasCreateDashboardPermission: true,
        onDashboardFilterChange,
        onDelete,
        onDuplicate,
    };

    const getPageTitle = () => wrapper.findWhere((node) => node.is(DotTypography) && node.props()['data-testid'] === 'page-title');
    const getDashboardSearch = () => wrapper.findWhere((node) => node.is(DotInputText) && node.props().id === 'dashboard-search');
    const getDotCards = () => wrapper.find(DotCard);
    const getDotMenu = () => wrapper.find(DotMenu);
    const getDotEmptyState = () => wrapper.find(DotEmptyState);
    const getDashboardActionContextMenu = () => wrapper.find(DashboardActionContextMenu);

    const mount = (props = defaultProps) => {
        wrapper = mountWithTheme(<DashboardCards {...props} />);
    };

    beforeEach(() => {
        const authenticator = {
            hasPermission: jest.fn(),
        };
        const getAngularServiceSpy = jest.spyOn(angularAccessor, 'default') as unknown as jest.SpyInstance;
        getAngularServiceSpy.mockReturnValue(authenticator);
    });

    afterEach(() => jest.resetAllMocks());

    it('should render correct elements', () => {
        mount();
        const pageTitle = getPageTitle();
        expect(pageTitle).toExist();
        expect(pageTitle.props().variant).toBe('h1');
        expect(pageTitle.props().children).toBe('Custom dashboards');

        const dashboardSearch = getDashboardSearch();
        expect(dashboardSearch).toExist();
        const dashboardSearchProps = dashboardSearch.props();
        expect(dashboardSearchProps.fullWidth).toBe(false);
        expect(dashboardSearchProps.id).toBe('dashboard-search');
        expect(dashboardSearchProps.name).toBe('dashboard-search');
        expect(dashboardSearchProps.placeholder).toBe('Filter by name');

        const cards = getDotCards();
        expect(cards.length).toBe(dashboards.length);

        const dotEmptyState = getDotEmptyState();
        expect(dotEmptyState).not.toExist();

        cards.forEach((card, index) => {
            const icon = card.findWhere((node) => node.is(DotIcon) && node.props()['data-testid'] === 'card-icon');
            expect(icon).toExist();
            expect(icon.props().iconId).toBe('dashboard');

            const dashboard = dashboards[index];
            expect(card.props().className).toBe('dashboard-card');
            const dotLink = card.find(DotLink);
            expect(dotLink).toExist();
            const dotLinkProps = dotLink.props();
            expect(dotLinkProps.ariaLabel).toBe(dashboard.title);
            expect(dotLinkProps.href).toBe(`#/dashboards/${dashboard.id}`);
            expect(dotLinkProps.underline).toBe('none');

            const folderLink = card.findWhere((node) => node.is(DotLink) && node.props()['data-testid'] === 'folder-link');
            expect(folderLink).not.toExist();

            const dotTypography = dotLink.find(DotTypography);
            expect(dotTypography).toExist();
            expect(dotTypography.props().variant).toBe('subtitle2');
            expect(dotTypography.props().children).toBe(dashboard.title);

            const contextMenuIconButton = card.find(DotIconButton);
            expect(contextMenuIconButton).toExist();
            expect(contextMenuIconButton.props().iconId).toBe('options');
        });

        const dotMenu = getDotMenu();
        expect(dotMenu).not.toExist();
    });

    it('should render links with correct href when folderId is defined', () => {
        const folder = foldersMock[0];
        mount({ ...defaultProps, folder });
        const cards = getDotCards();
        cards.forEach((card, index) => {
            const dashboard = dashboards[index];
            const dotLink = card.findWhere((node) => node.is(DotLink) && node.props()['data-testid'] === 'dashboard-link');
            expect(dotLink.props().href).toBe(`#/folders/${folder.id}/dashboards/${dashboard.id}`);

            const folderLink = card.findWhere((node) => node.is(DotLink) && node.props()['data-testid'] === 'folder-link');
            expect(folderLink).toExist();
            const folderLinkProps = folderLink.props();
            expect(folderLinkProps.ariaLabel).toBe(folder.title);
            expect(folderLinkProps.href).toBe(`#/folders/${folder.id}/${DEFAULT_FOLDER_TAB}`);
            expect(folderLinkProps.underline).toBe('none');

            const folderLinkTypography = folderLink.find(DotTypography);
            expect(folderLinkTypography).toExist();
            expect(folderLinkTypography.props().variant).toBe('body1');
            expect(folderLinkTypography.props().children).toBe(folder.title);
        });
    });

    it('should trigger correct handler upon filter change', () => {
        const searchText = 'test';
        mount();
        const dashboardSearch = getDashboardSearch();
        dashboardSearch.invoke('onChange')({ target: { value: searchText } });
        expect(onDashboardFilterChange).toHaveBeenCalledWith(searchText);
    });

    it('should render DotEmptyState component when no dashboards', () => {
        mount({ ...defaultProps, dashboards: [] });
        const dotEmptyState = getDotEmptyState();
        expect(dotEmptyState).toExist();
        const dotEmptyStateProps = dotEmptyState.props();
        expect(dotEmptyStateProps.className).toBe('dashboard-cards-empty-state');
    });

    it('should set correct anchor and dashboard props when context menu icon button has been clicked', () => {
        mount();

        const initialContextMenuProps = getDashboardActionContextMenu().props();
        expect(initialContextMenuProps.anchorEl).toBeUndefined();
        expect(initialContextMenuProps.dashboard).toBeNull();

        const cards = getDotCards();
        const iconButton = cards.at(0).find(DotIconButton);
        const element = document.createElement('div');
        iconButton.invoke('onClick')?.({ currentTarget: element, stopPropagation } as never);

        const contextMenuProps = getDashboardActionContextMenu().props();
        expect(contextMenuProps.anchorEl).toStrictEqual(element);
        expect(contextMenuProps.dashboard).toStrictEqual(dashboards[0]);
    });

    it('should trigger correct handler when onDelete event has been invoked', () => {
        mount();
        const dashboardId = '123';
        const contextMenu = getDashboardActionContextMenu();
        contextMenu.invoke('onDelete')(dashboardId);
        expect(onDelete).toHaveBeenCalledWith(dashboardId);
    });

    it('should trigger correct handler when onDuplicate event has been invoked', () => {
        mount();
        const dashboard = dashboards[0];
        const contextMenu = getDashboardActionContextMenu();
        contextMenu.invoke('onDuplicate')(dashboard);
        expect(onDuplicate).toHaveBeenCalledWith(dashboard);
    });

    it('should set active dashboard to null when onLeave event has been invoked', () => {
        mount();
        const activeDashboard = dashboards[0];

        const cards = getDotCards();
        const iconButton = cards.at(0).find(DotIconButton);
        iconButton.invoke('onClick')?.({ currentTarget: document.createElement('div'), stopPropagation } as never);

        expect(getDashboardActionContextMenu().props().dashboard).toStrictEqual(activeDashboard);

        const contextMenu = getDashboardActionContextMenu();
        contextMenu.invoke('onLeave')();
        expect(getDashboardActionContextMenu().props().dashboard).toStrictEqual(null);
    });

    it('should trigger correct handler when filtering', () => {
        mount();
        const dashboardId = '123';
        const contextMenu = getDashboardActionContextMenu();
        contextMenu.invoke('onDelete')(dashboardId);
        expect(onDelete).toHaveBeenCalledWith(dashboardId);
    });
});
