import React from 'react';
import { mountWithTheme, ReactWrapper } from '@xlr-ui/tests/unit/testing-utils';
import { dashboardsMock } from '../../../mocks';
import { DotConfirmationDialog, DotIcon } from '@digital-ai/dot-components';
import { DashboardDeleteDialogContent } from './dashboard-delete-dialog-content.component';
import { DashboardDuplicateDialogContent } from './dashboard-duplicate-dialog-content.component';
import { DashboardDialog, DashboardDialogProps, LENGTH_ERROR_MESSAGE } from './dashboard-dialog.component';

describe('DashboardDialog', () => {
    let wrapper: ReactWrapper;
    const dashboard = dashboardsMock[0];
    const dialogTitle = 'My dialog title';
    const onDialogClose = jest.fn();
    const onDialogSubmit = jest.fn();

    const defaultProps: DashboardDialogProps = {
        dashboard,
        dashboardActionType: 'DELETE',
        dialogTitle,
        onDialogClose,
        onDialogSubmit,
    };
    const getDotConfirmationDialog = () => wrapper.find(DotConfirmationDialog);
    const getDashboardDeleteDialogContent = () => wrapper.find(DashboardDeleteDialogContent);
    const getDashboardDuplicateDialogContent = () => wrapper.find(DashboardDuplicateDialogContent);
    const getDuplicateStartIcon = () => wrapper.findWhere((node) => node.is(DotIcon) && node.props()['data-testid'] === 'duplicate-start-icon');
    const getDeleteStartIcon = () => wrapper.findWhere((node) => node.is(DotIcon) && node.props()['data-testid'] === 'delete-start-icon');

    const mount = (props = defaultProps) => {
        wrapper = mountWithTheme(<DashboardDialog {...props} />);
    };

    afterEach(() => jest.resetAllMocks());

    it('should render correct elements with the DELETE action', () => {
        mount();

        const dotConfirmationDialog = getDotConfirmationDialog();
        expect(dotConfirmationDialog).toExist();
        const dotConfirmationDialogProps = dotConfirmationDialog.props();
        expect(dotConfirmationDialogProps.className).toBe('dashboard-confirmation-dialog');
        expect(dotConfirmationDialogProps.open).toBe(true);
        expect(dotConfirmationDialogProps.submitButtonProps.label).toBe('Delete dashboard');
        expect(dotConfirmationDialogProps.submitButtonProps.type).toBe('destructive');
        expect(dotConfirmationDialogProps.title).toBe(dialogTitle);

        const dashboardDeleteDialogContent = getDashboardDeleteDialogContent();
        expect(dashboardDeleteDialogContent).toExist();
        expect(dashboardDeleteDialogContent.props().dashboard).toStrictEqual(dashboard);

        const deleteStartIcon = getDeleteStartIcon();
        expect(deleteStartIcon).toExist();
        expect(deleteStartIcon.props().iconId).toBe('delete');
    });

    it('should render correct elements with the DUPLICATE action', () => {
        mount({ ...defaultProps, dashboardActionType: 'DUPLICATE' });

        const dotConfirmationDialog = getDotConfirmationDialog();
        expect(dotConfirmationDialog).toExist();
        const dotConfirmationDialogProps = dotConfirmationDialog.props();
        expect(dotConfirmationDialogProps.className).toBe('dashboard-confirmation-dialog');
        expect(dotConfirmationDialogProps.open).toBe(true);
        expect(dotConfirmationDialogProps.submitButtonProps.label).toBe('Duplicate dashboard');
        expect(dotConfirmationDialogProps.submitButtonProps.disabled).toBe(false);
        expect(dotConfirmationDialogProps.submitButtonProps.type).toBe('primary');
        expect(dotConfirmationDialogProps.title).toBe(dialogTitle);

        const dashboardDuplicateDialogContent = getDashboardDuplicateDialogContent();
        expect(dashboardDuplicateDialogContent).toExist();
        const dashboardDuplicateDialogContentProps = dashboardDuplicateDialogContent.props();
        expect(dashboardDuplicateDialogContentProps.dashboard).toStrictEqual(dashboard);
        expect(dashboardDuplicateDialogContentProps.errorMessage).toBeUndefined();
        expect(dashboardDuplicateDialogContentProps.title).toBe(`${dashboard.title} (copy)`);

        const duplicateStartIcon = getDuplicateStartIcon();
        expect(duplicateStartIcon).toExist();
        expect(duplicateStartIcon.props().iconId).toBe('duplicate');
    });

    it('should render DashboardDuplicateDialogContent component with error message', () => {
        mount({ ...defaultProps, dashboardActionType: 'DUPLICATE', dashboard: { ...dashboard, title: dashboard.title.repeat(200) } });
        const dashboardDuplicateDialogContentProps = getDashboardDuplicateDialogContent().props();
        expect(dashboardDuplicateDialogContentProps.errorMessage).toBe(LENGTH_ERROR_MESSAGE);

        const dotConfirmationDialog = getDotConfirmationDialog();
        expect(dotConfirmationDialog.props().submitButtonProps.disabled).toBe(true);
    });

    it('should set correct title on the DashboardDuplicateDialogContent when the title changes on the component', () => {
        const newTitle = 'new title';
        mount({ ...defaultProps, dashboardActionType: 'DUPLICATE' });
        const dashboardDuplicateDialogContent = getDashboardDuplicateDialogContent();
        dashboardDuplicateDialogContent.invoke('onNameChange')(newTitle);
        expect(getDashboardDuplicateDialogContent().props().title).toBe(newTitle);
    });

    it('should execute correct handler when onSubmit is invoked from duplicate dashboard dialog', () => {
        const newTitle = 'new title';
        mount({ ...defaultProps, dashboardActionType: 'DUPLICATE' });
        const dashboardDuplicateDialogContent = getDashboardDuplicateDialogContent();
        dashboardDuplicateDialogContent.invoke('onNameChange')(newTitle);
        const dotConfirmationDialog = getDotConfirmationDialog();
        dotConfirmationDialog.invoke('onSubmit')(newTitle);
        expect(onDialogSubmit).toHaveBeenLastCalledWith(newTitle);
    });

    it('should execute correct handler when onSubmit is invoked from delete dashboard dialog', () => {
        mount();
        const dotConfirmationDialog = getDotConfirmationDialog();
        dotConfirmationDialog.invoke('onSubmit')();
        expect(onDialogSubmit).toHaveBeenLastCalledWith();
    });

    describe('empty duplicate title', () => {
        const assertDisabledSubmitForText = (title: string) => {
            mount({ ...defaultProps, dashboardActionType: 'DUPLICATE' });

            const dashboardDuplicateDialogContent = getDashboardDuplicateDialogContent();
            dashboardDuplicateDialogContent.invoke('onNameChange')(title);

            const dialogProps = getDotConfirmationDialog().props();
            expect(dialogProps.submitButtonProps?.disabled).toBe(true);
        };

        it('should disable duplicate submit button when input text is empty', () => {
            assertDisabledSubmitForText('');
        });

        it('should disable duplicate submit button when input text has only spaces', () => {
            assertDisabledSubmitForText('  ');
        });
    });
});
