import React, { useState } from 'react';
import { DotMenu, MenuItemProps } from '@digital-ai/dot-components';
import { getDuplicatedDashboardFromOriginal, mapDashboardActions } from '../../helper';
import { Dashboard, DashboardActionId, DashboardDialogDataProps } from '../../types';
import { DashboardDialog } from './list/dashboard-dialog.component';

export interface DashboardActionContextMenuProps {
    anchorEl: HTMLElement | undefined;
    dashboard: Dashboard | null;
    hasCreateDashboardPermission: boolean;
    hasDeleteDashboardPermission: boolean;
    onDelete: (dashboardId: string) => void;
    onDuplicate: (dashboard: Dashboard) => void;
    onLeave: () => void;
}

export const DashboardActionContextMenu = ({
    anchorEl,
    dashboard,
    hasCreateDashboardPermission,
    hasDeleteDashboardPermission,
    onDelete,
    onDuplicate,
    onLeave,
}: DashboardActionContextMenuProps) => {
    const [dialogData, setDialogData] = useState<DashboardDialogDataProps | undefined>(undefined);
    const isMenuRendered = dashboard && anchorEl;
    const contextMenuItems = mapDashboardActions([
        { actionId: 'DUPLICATE', disabled: false },
        { actionId: 'DELETE', disabled: false },
    ]);

    const getContextMenuItems = (menuItems: MenuItemProps[]): MenuItemProps[] => {
        return menuItems.map((item) => {
            const mappedItem = { ...item };
            if (mappedItem.key === 'DUPLICATE') {
                mappedItem.disabled = !hasCreateDashboardPermission;
            } else if (mappedItem.key === 'DELETE') {
                mappedItem.disabled = !hasDeleteDashboardPermission;
            }
            return mappedItem;
        });
    };

    const handleMenuItemClick = (itemKey: DashboardActionId, currentDashboard: Dashboard) => {
        if (itemKey === 'DELETE') {
            setDialogData({
                dashboard: currentDashboard,
                dialogTitle: 'Delete dashboard',
                actionId: 'DELETE',
            });
        } else if (itemKey === 'DUPLICATE') {
            setDialogData({
                dashboard: currentDashboard,
                dialogTitle: 'Duplicate dashboard',
                actionId: 'DUPLICATE',
            });
        }
    };

    const handleDialogClose = () => {
        setDialogData(undefined);
    };

    const handleDialogSubmit = (duplicateTitle?: string) => {
        const { actionId, dashboard: currentDashboard } = dialogData as DashboardDialogDataProps;
        if (!currentDashboard.id) return;
        if (actionId === 'DELETE') {
            onDelete(currentDashboard.id);
        } else if (actionId === 'DUPLICATE' && duplicateTitle) {
            onDuplicate(getDuplicatedDashboardFromOriginal(currentDashboard, duplicateTitle));
        }
        setDialogData(undefined);
    };

    return (
        <>
            {isMenuRendered && (
                <DotMenu
                    anchorEl={anchorEl}
                    dense
                    disablePortal
                    id="dashboard-context-menu"
                    menuItems={getContextMenuItems(contextMenuItems)}
                    menuPlacement="bottom-end"
                    onLeave={onLeave}
                    onSelect={(_event, _menuId, itemKey) => handleMenuItemClick(itemKey as DashboardActionId, dashboard)}
                    open={true}
                />
            )}
            {dialogData && (
                <DashboardDialog
                    dashboard={dialogData.dashboard}
                    dashboardActionType={dialogData.actionId}
                    dialogTitle={dialogData.dialogTitle}
                    onDialogClose={handleDialogClose}
                    onDialogSubmit={handleDialogSubmit}
                />
            )}
        </>
    );
};
