import React from 'react';
import { DotInputText, DotTypography } from '@digital-ai/dot-components';
import { dashboardsMock } from '../../../mocks';
import { mountWithTheme, ReactWrapper } from '@xlr-ui/tests/unit/testing-utils';
import { DashboardDuplicateDialogContent, DashboardDuplicateDialogContentProps } from './dashboard-duplicate-dialog-content.component';

describe('DashboardDuplicateDialogContent', () => {
    let wrapper: ReactWrapper;
    const dashboard = dashboardsMock[0];
    const title = 'My title';
    const onNameChange = jest.fn();

    const defaultProps: DashboardDuplicateDialogContentProps = {
        dashboard,
        onNameChange,
        title,
    };

    const getDotTypography = () => wrapper.findWhere((node) => node.is(DotTypography) && node.props()['data-testid'] === 'duplicate-dashboard-typography');
    const getDotInputText = () => wrapper.find(DotInputText);

    const mount = (props = defaultProps) => {
        wrapper = mountWithTheme(<DashboardDuplicateDialogContent {...props} />);
    };

    it('should render correct elements', () => {
        mount();
        const typography = getDotTypography();
        expect(typography).toExist();
        const typographyProps = getDotTypography().props();
        expect(typographyProps.variant).toBe('body1');
        expect(typography).toHaveText(`You are about to Duplicate "${dashboard.title}".Provide a new name for your duplicated dashboard.`);

        const dotInputText = getDotInputText();
        expect(dotInputText).toExist();
        const dotInputTextProps = dotInputText.props();
        expect(dotInputTextProps.autoFocus).toBe(true);
        expect(dotInputTextProps.defaultValue).toBe(title);
        expect(dotInputTextProps.error).toBe(false);
        expect(dotInputTextProps.helperText).toBe(undefined);
        expect(dotInputTextProps.id).toBe('dashboard-name');
        expect(dotInputTextProps.label).toBe('Dashboard name');
        expect(dotInputTextProps.maxLength).toBe(255);
        expect(dotInputTextProps.name).toBe('dashboard-name');
        expect(dotInputTextProps.persistentLabel).toBe(true);
    });

    it('should trigger correct handler when onChange has been executed on the input text', () => {
        mount();
        const newTitle = 'my new title';
        const event = { target: { value: newTitle } };
        const dotInputText = getDotInputText();
        dotInputText.invoke('onChange')?.(event as never);
        expect(onNameChange).toHaveBeenCalledWith(newTitle);
    });

    it('should have correct props when error message is defined', () => {
        const errorMessage = 'going to accident';
        mount({ ...defaultProps, errorMessage });
        const dotInputTextProps = getDotInputText().props();
        expect(dotInputTextProps.error).toBe(true);
        expect(dotInputTextProps.helperText).toBe(errorMessage);
    });
});
