import { call, put } from 'redux-saga/effects';
import { SagaIterator } from 'redux-saga';
import getAngularService from '@xlr-ui/app/features/common/services/angular-accessor';
import { navigation } from '@xlr-ui/app/features/main-navigation/ducks/navigation.reducer';
import { NewDashboardSidebarProvider } from './xlr-new-dashboard-provider';

const { setBackItem, updateBreadcrumbs, setSidebarMenu, setLastBreadcrumb } = navigation.actions;

describe('NewDashboardSidebarProvider', () => {
    const newDashboardSidebarProvider = new NewDashboardSidebarProvider();

    describe('isSupported', () => {
        it('should be false for not supported path', () => {
            expect(newDashboardSidebarProvider.isSupported('/dashboards/')).toBe(false);
        });

        it('should return true for paths that start with /dashboards/new', () => {
            expect(newDashboardSidebarProvider.isSupported('/dashboards/new')).toBe(true);
        });
    });

    describe('provide', () => {
        const UiExtensionsService = {
            getMainMenuItems: jest.fn(),
        };

        const mainMenuItems = [
            {
                id: 'xlrelease.menu.Reports',
                items: [
                    {
                        icon: 'dashboard',
                        id: 'xlrelease.menu.Dashboards',
                        items: [
                            {
                                icon: 'report',
                                label: 'Summary',
                                pathSuffix: 'reports',
                                properties: {
                                    permissions: 'reports#view,all#audit',
                                },
                                uri: '',
                                weight: 10,
                                permitted: true,
                            },
                            {
                                icon: 'dashboard',
                                label: 'Custom dashboards',
                                pathSuffix: 'dashboards/list',
                                properties: {},
                                uri: '',
                                weight: 50,
                                permitted: true,
                            },
                        ],
                        label: 'Dashboards',
                        permissions: [],
                        resources: [],
                        uri: 'dashboards',
                        weight: 10,
                        permitted: true,
                    },
                ],
            },
        ];

        const breadcrumbs = [{ text: 'Dashboards', href: `/dashboards` }, { text: 'Custom dashboards', href: `/dashboards/list` }, { text: 'New dashboard' }];

        const dashboardSidebar = [
            {
                text: 'Summary',
                startIconId: 'report',
                pathSuffix: 'reports',
                selected: false,
            },
            {
                text: 'Custom dashboards',
                startIconId: 'dashboard',
                pathSuffix: 'dashboards/list',
                selected: true,
            },
        ];

        it('should call proper sagas', () => {
            const gen: SagaIterator = newDashboardSidebarProvider.provide();
            expect(gen.next().value).toMatchObject(call(getAngularService, 'UiExtensionsService'));
            expect(gen.next(UiExtensionsService).value).toMatchObject(call([UiExtensionsService, UiExtensionsService.getMainMenuItems]));

            expect(gen.next(mainMenuItems).value).toMatchObject(put(setSidebarMenu({ sidebarMenu: dashboardSidebar, withFolder: false })));
            expect(gen.next().value).toMatchObject(put(setLastBreadcrumb(undefined)));
            expect(gen.next().value).toMatchObject(put(updateBreadcrumbs(breadcrumbs)));
            expect(gen.next().value).toMatchObject(
                put(
                    setBackItem({
                        pathSuffix: 'dashboards/list',
                        title: 'Custom dashboards',
                        text: 'Custom dashboards',
                    }),
                ),
            );
            expect(gen.next().done).toBe(true);
        });
    });
});
