package com.xebialabs.xlrelease.plugins.dashboard.repository.sql.persistence

import com.xebialabs.xlplatform.utils.ResourceManagement.using
import com.xebialabs.xlrelease.db.sql.SqlBuilder.Dialect
import com.xebialabs.xlrelease.db.sql.SqlWithParameters
import com.xebialabs.xlrelease.db.sql.transaction.IsTransactional
import com.xebialabs.xlrelease.plugins.dashboard.domain.Dashboard
import com.xebialabs.xlrelease.plugins.dashboard.repository.sql.persistence.DashboardSchema.{COLUMN_LENGTH_TITLE, DASHBOARDS}
import com.xebialabs.xlrelease.plugins.dashboard.repository.sql.persistence.DashboardSqlBuilder.{STMT_DASHBOARD_SELECT, alias}
import com.xebialabs.xlrelease.repository.Ids.getName
import com.xebialabs.xlrelease.repository.sql.persistence.CiId.{CiId, RichCiId}
import com.xebialabs.xlrelease.repository.sql.persistence.Schema.FOLDERS
import com.xebialabs.xlrelease.repository.sql.persistence.Utils.{RichStringAsTruncatable, params}
import com.xebialabs.xlrelease.repository.sql.persistence.{CiUid, PersistenceSupport, SecurablePersistence}
import com.xebialabs.xlrelease.serialization.json.utils.CiSerializerHelper
import com.xebialabs.xlrelease.utils.FolderId
import org.springframework.beans.factory.annotation.{Autowired, Qualifier}
import org.springframework.dao.DuplicateKeyException
import org.springframework.jdbc.core.{JdbcTemplate, RowMapper}
import org.springframework.stereotype.Repository

import java.sql.ResultSet
import scala.jdk.CollectionConverters._
import scala.util.Try


@Repository
@IsTransactional
class DashboardPersistence @Autowired()(@Qualifier("xlrRepositoryJdbcTemplate") implicit val jdbcTemplate: JdbcTemplate,
                                        @Qualifier("xlrRepositorySqlDialect") implicit val dialect: Dialect,
                                        val securablePersistence: SecurablePersistence)
  extends PersistenceSupport {

  private val STMT_EXISTS_DASHBOARD_BY_ID = s"SELECT COUNT(*) FROM ${DASHBOARDS.TABLE} WHERE ${DASHBOARDS.ID} = :dashboardId"

  def exists(dashboardId: CiId): Boolean = sqlQuery(STMT_EXISTS_DASHBOARD_BY_ID, params("dashboardId" -> getName(dashboardId)), _.getInt(1) > 0).head

  private val STMT_GET_UID_BY_ID = s"SELECT ${DASHBOARDS.CI_UID} FROM ${DASHBOARDS.TABLE} WHERE ${DASHBOARDS.ID} = :dashboardId"

  def getUid(dashboardId: CiId): Option[CiUid] =
    sqlQuery(STMT_GET_UID_BY_ID, params("dashboardId" -> getName(dashboardId.normalized)), rs => CiUid(rs.getInt(DASHBOARDS.CI_UID))).headOption

  private val STMT_INSERT_DASHBOARD =
    s"""|INSERT INTO ${DASHBOARDS.TABLE}
        |   ( ${DASHBOARDS.CI_UID}
        |   , ${DASHBOARDS.ID}
        |   , ${DASHBOARDS.PARENT_ID}
        |   , ${DASHBOARDS.OWNER}
        |   , ${DASHBOARDS.TITLE}
        |   , ${DASHBOARDS.CONTENT}
        |   )
        | VALUES
        |   ( :ciUid
        |   , :id
        |   , :parentId
        |   , :owner
        |   , :title
        |   , :content
        |   )
        """.stripMargin

  def insert(dashboard: Dashboard): CiUid = {
    val ciUid = securablePersistence.insert()
    val content = serialize(dashboard)
    try {
      sqlExecWithContent(STMT_INSERT_DASHBOARD, params(
        "ciUid" -> ciUid,
        "id" -> getName(dashboard.getId.normalized),
        "parentId" -> Option(dashboard.getParentId).map(id => getName(id.normalized)).orNull,
        "owner" -> dashboard.getOwner,
        "title" -> dashboard.getTitle.truncate(COLUMN_LENGTH_TITLE)
      ), "content" -> content,
        _ => ())
    } catch {
      case ex: DuplicateKeyException => throw new IllegalArgumentException(s"Dashboard with ID '${dashboard.getId}' already exists", ex)
    }
    ciUid
  }

  private val STMT_UPDATE_DASHBOARD =
    s"""|UPDATE ${DASHBOARDS.TABLE}
        | SET
        |  ${DASHBOARDS.TITLE} = :newTitle,
        |  ${DASHBOARDS.PARENT_ID} = :newParentId,
        |  ${DASHBOARDS.OWNER} = :newOwner,
        |  ${DASHBOARDS.CONTENT} = :content
        | WHERE
        |  ${DASHBOARDS.ID} = :dashboardId
        """.stripMargin

  def update(dashboard: Dashboard): Unit = {
    val content = serialize(dashboard)
    sqlExecWithContent(STMT_UPDATE_DASHBOARD, params(
      "dashboardId" -> getName(dashboard.getId),
      "newParentId" -> Option(dashboard.getParentId).map(id => getName(id.normalized)).orNull,
      "newOwner" -> dashboard.getOwner,
      "newTitle" -> dashboard.getTitle.truncate(COLUMN_LENGTH_TITLE)
    ), "content" -> content,
      checkCiUpdated(dashboard.getId)
    )
  }

  private val STMT_DELETE_DASHBOARD_BY_ID =
    s"""|DELETE FROM ${DASHBOARDS.TABLE}
        | WHERE ${DASHBOARDS.ID} = :dashboardId""".stripMargin

  def delete(dashboardId: CiId): Try[Boolean] =
    sqlExec(STMT_DELETE_DASHBOARD_BY_ID, params("dashboardId" -> getName(dashboardId.normalized)), ps => Try(ps.execute()))

  private val STMT_DELETE_DASHBOARDS_BY_PARENT_ID =
    s"""|DELETE FROM ${DASHBOARDS.TABLE}
        | WHERE ${DASHBOARDS.PARENT_ID} = :parentId""".stripMargin

  def deleteByParentId(parentId: CiId): Try[Boolean] =
    sqlExec(STMT_DELETE_DASHBOARDS_BY_PARENT_ID, params("parentId" -> getName(parentId.normalized)), ps => Try(ps.execute()))

  private val STMT_FIND_DASHBOARD_BY_ID =
    s"""|$STMT_DASHBOARD_SELECT
        | WHERE $alias.${DASHBOARDS.ID} = :dashboardId""".stripMargin

  def findById(dashboardId: CiId): Option[DashboardRow] = {
    sqlQuery(STMT_FIND_DASHBOARD_BY_ID, params("dashboardId" -> getName(dashboardId.normalized)), dashboardRowMapper).headOption
  }

  def findByQuery(sqlWithParameters: SqlWithParameters): Seq[DashboardRow] = {
    val (sql, params) = sqlWithParameters
    jdbcTemplate.query[DashboardRow](sql, dashboardRowMapper, params: _*).asScala.toSeq
  }

  private val dashboardRowMapper: RowMapper[DashboardRow] = (rs: ResultSet, _: Int) => {
    using(rs.getBinaryStream(DASHBOARDS.CONTENT)) { contentStream =>
      val content = decompress(contentStream)
      DashboardRow(
        rs.getString(DASHBOARDS.ID),
        (FolderId.Root / rs.getString(FOLDERS.FOLDER_PATH) / rs.getString(FOLDERS.FOLDER_ID)).absolute,
        content
      )
    }
  }

  private def serialize(dashboard: Dashboard): String = {
    val content = CiSerializerHelper.serialize(dashboard)
    content
  }
}
