package com.xebialabs.xlrelease.plugins.dashboard.service

import com.xebialabs.deployit.plugin.api.udm.ConfigurationItem
import com.xebialabs.deployit.security.permission.Permission
import com.xebialabs.deployit.security.{Permissions, Role, RoleService}
import com.xebialabs.xlrelease.api.internal.EffectiveSecurityDecorator.EFFECTIVE_SECURITY
import com.xebialabs.xlrelease.api.internal.{CachedInternalMetadataDecorator, DecoratorCache, EffectiveSecurity}
import com.xebialabs.xlrelease.plugins.dashboard.domain.Dashboard
import com.xebialabs.xlrelease.repository.sql.persistence.ListExtensions
import com.xebialabs.xlrelease.security.XLReleasePermissions
import org.springframework.beans.factory.annotation.Autowired
import org.springframework.security.core.Authentication
import org.springframework.stereotype.Component

import java.util
import scala.collection.mutable
import scala.jdk.CollectionConverters._

case class DashboardSecurityDecoratorCache(authentication: Authentication, roles: Seq[Role])

@Component
class DashboardSecurityDecorator @Autowired()(roleService: RoleService,
                                              dashboardSecurity: DashboardSecurity) extends CachedInternalMetadataDecorator[Dashboard, DashboardSecurityDecoratorCache] {

  lazy val ownerEffectiveSecurity: EffectiveSecurity = new EffectiveSecurity(
    Set(XLReleasePermissions.VIEW_DASHBOARD, XLReleasePermissions.EDIT_DASHBOARD).map(_.getPermissionName).asJava,
    Set.empty[String].asJava
  )

  override def isApplicableTo(ci: ConfigurationItem): Boolean = ci.isInstanceOf[Dashboard]

  override val name: String = EFFECTIVE_SECURITY

  override def decorate(cis: util.Collection[Dashboard], cache: DecoratorCache[DashboardSecurityDecoratorCache]): Unit = {
    cache.computeIfAbsent(() => {
      val authentication: Authentication = Permissions.getAuthentication
      val userRoles: Seq[Role] = if (authentication != null) roleService.getRolesFor(authentication).asScala.toSeq else Seq.empty
      DashboardSecurityDecoratorCache(authentication, userRoles)
    }) match {
      case DashboardSecurityDecoratorCache(authentication, userRoles) => {
        cis.asScala.foreach { ci =>
          val permissions = dashboardSecurity.getPermissions(ci)
          ci.setRoleViewers(getRolesNamesWithPermission(permissions, XLReleasePermissions.VIEW_DASHBOARD).asJavaMutable())
          ci.setRoleEditors(getRolesNamesWithPermission(permissions, XLReleasePermissions.EDIT_DASHBOARD).asJavaMutable())

          if (authentication != null) {
            if (ci.getOwner == authentication.getName) {
              ci.get$metadata.put(EFFECTIVE_SECURITY, ownerEffectiveSecurity)
            } else {
              ci.get$metadata.put(EFFECTIVE_SECURITY, getSecurityForRoles(ci, permissions, authentication.getName, userRoles))
            }
          }
        }
      }
    }
  }


  private def getSecurityForRoles(dashboard: Dashboard, permissions: mutable.Map[Role, mutable.Set[Permission]],
                                  user: String, userRoles: Seq[Role]): EffectiveSecurity = {
    val myPermissions = permissions.view.filterKeys { role =>
      userRoles.contains(role) ||
        role.getPrincipals.contains(user) ||
        role.getRoles.asScala.exists(roleName => userRoles.exists(userRole => userRole.getName == roleName))
    }

    val effectiveSecurity: EffectiveSecurity = new EffectiveSecurity
    effectiveSecurity.setPermissions(myPermissions.values.flatMap(_.map(_.getPermissionName)).toSet.asJava)
    effectiveSecurity.setTeams(myPermissions.keys.map(_.getName).toSet.asJava)
    effectiveSecurity
  }

  private def getRolesNamesWithPermission(permissions: mutable.Map[Role, mutable.Set[Permission]], permission: Permission): Seq[String] = {
    permissions.filter { case (_, rolePermissions) => rolePermissions.contains(permission) }.keys.map(_.getName).toSeq
  }

}
