import { call, put } from 'redux-saga/effects';
import { SagaIterator } from 'redux-saga';
import { DashboardSidebarProvider } from './xlr-dashboard-provider';
import getAngularService from '../../../../../../../../../core/xlr-ui/app/features/common/services/angular-accessor';
import { navigation } from '../../../../../../../../../core/xlr-ui/app/features/main-navigation/ducks/navigation.reducer';
import { BreadcrumbItem } from '../../../../../../../../../core/xlr-ui/app/features/tasks/types/angular';

const { setBackItem, updateBreadcrumbs, setSidebarMenu, setLastBreadcrumb } = navigation.actions;

describe('DashboardSidebarProvider', () => {
    const dashboardSidebarProvider = new DashboardSidebarProvider();

    describe('isSupported', () => {
        it('should be false for not supported path', () => {
            expect(dashboardSidebarProvider.isSupported('/dashboards/Applications/')).toBe(false);
        });

        it('should return true for paths that start with /dashboards/Applications/Dashboard', () => {
            expect(dashboardSidebarProvider.isSupported('/dashboards/Applications/Dashboard123')).toBe(true);
        });
    });

    describe('provide', () => {
        const xlrDashboardService = {
            loadConfiguration: jest.fn(),
        };

        const dashboard = {
            title: 'my dashboard',
            id: 'Applications/Dashboard077f84862b9e4849a867fde300184e91',
        };

        const breadcrumbs: BreadcrumbItem[] = [
            { text: 'Reports', href: `/dashboards` },
            { text: 'Custom dashboards', href: `/dashboards/list` },
            { text: dashboard.title, href: `/dashboards/${dashboard.id}` },
        ];

        it('should call proper sagas when NOT in dashboard details', () => {
            const gen: SagaIterator = dashboardSidebarProvider.provide(`/dashboards/${dashboard.id}`);
            expect(gen.next().value).toMatchObject(call(getAngularService, 'xlrelease.XlrDashboardService'));
            expect(gen.next(xlrDashboardService).value).toMatchObject(call([xlrDashboardService, xlrDashboardService.loadConfiguration], dashboard.id));

            expect(gen.next({ data: dashboard }).value).toMatchObject(put(setLastBreadcrumb(undefined)));
            expect(gen.next().value).toMatchObject(put(updateBreadcrumbs(breadcrumbs)));
            expect(gen.next().value).toMatchObject(put(setSidebarMenu({ sidebarMenu: [], withFolder: false })));
            expect(gen.next().value).toMatchObject(
                put(
                    setBackItem({
                        pathSuffix: 'dashboards/list',
                        title: 'Custom dashboards',
                        text: 'Custom dashboards',
                    }),
                ),
            );
            expect(gen.next().done).toBe(true);
        });

        it('should call proper sagas when in dashboard details', () => {
            const gen: SagaIterator = dashboardSidebarProvider.provide(`/dashboards/${dashboard.id}/Tilebc7ada8f05ec48368fbbb0bff7a64a75/details`);
            expect(gen.next().value).toMatchObject(call(getAngularService, 'xlrelease.XlrDashboardService'));
            expect(gen.next(xlrDashboardService).value).toMatchObject(call([xlrDashboardService, xlrDashboardService.loadConfiguration], dashboard.id));

            expect(gen.next({ data: dashboard }).value).toMatchObject(put(setLastBreadcrumb(undefined)));
            expect(gen.next().value).toMatchObject(put(updateBreadcrumbs([...breadcrumbs, { text: 'Details' }])));
            expect(gen.next().value).toMatchObject(put(setSidebarMenu({ sidebarMenu: [], withFolder: false })));
            expect(gen.next().value).toMatchObject(
                put(
                    setBackItem({
                        pathSuffix: `/dashboards/${dashboard.id}`,
                        title: dashboard.title,
                        text: dashboard.title,
                    }),
                ),
            );
            expect(gen.next().done).toBe(true);
        });
    });
});
