package com.xebialabs.xlrelease.plugins.dashboard.api.internal

import com.fasterxml.jackson.databind.ObjectMapper
import com.xebialabs.deployit.plugin.api.reflect.Descriptor
import com.xebialabs.xlrelease.plugins.dashboard.domain.TileScope
import com.xebialabs.xlrelease.plugins.dashboard.service.{DashboardSecurity, TileService}
import com.xebialabs.xlrelease.plugins.dashboard.views.{TileData, TileView}
import com.xebialabs.xlrelease.repository.ConfigurationRepository
import com.xebialabs.xlrelease.repository.Ids.getParentId
import org.springframework.beans.factory.annotation.Autowired
import org.springframework.stereotype.Controller

import java.util.{List => JList}
import jakarta.ws.rs._
import jakarta.ws.rs.core.{Context, MediaType, UriInfo}
import scala.jdk.CollectionConverters._

@Path("/tiles")
@Controller
@Produces(Array(MediaType.APPLICATION_JSON))
@Consumes(Array(MediaType.APPLICATION_JSON))
class TileResource @Autowired()(val dashboardSecurity: DashboardSecurity,
                                val tilesService: TileService,
                                implicit val configurationRepository: ConfigurationRepository,
                                implicit val objectMapper: ObjectMapper) {
  @GET
  @Path("/tile-types")
  def getTileTypes(@QueryParam("scope") @DefaultValue("RELEASE") scope: TileScope): JList[Descriptor] = tilesService.getTileTypes(scope).asJava

  @GET
  @Path("/{tileId:.*Tile[^/-]*}/data")
  def getTileData(@PathParam("tileId") tileId: String,
                  @Context info: UriInfo): TileData = {
    dashboardSecurity.checkView(getParentId(tileId))
    TileData(tilesService.executeTileDataScript(tileId, getQueryParameters(info)))
  }

  @GET
  @Path("{tileId:.*Tile[^/-]*}")
  def getTile(@PathParam("tileId") tileId: String): TileView = {
    dashboardSecurity.checkView(getParentId(tileId))
    tilesService.findTileById(tileId)
  }

  @PUT
  @Path("{tileId:.*Tile[^/-]*}")
  def updateTile(@PathParam("tileId") tileId: String, tile: TileView): TileView = {
    dashboardSecurity.checkEdit(getParentId(tileId))
    tile.setId(tileId)
    import com.xebialabs.xlrelease.plugins.dashboard.views._
    tilesService.updateTile(viewToTile(tile))
  }

  private[internal] def getQueryParameters(info: UriInfo): Map[String, Any] = info.getQueryParameters.asScala.map {
    case (k, v) => k -> info.getQueryParameters.getFirst(k)
  }.toMap

}
