package com.xebialabs.xlrelease.plugins.dashboard.service

import com.xebialabs.deployit.plugin.api.udm.ConfigurationItem
import com.xebialabs.xlrelease.config.CacheManagementConstants.TILE_CACHE_MANAGER
import com.xebialabs.xlrelease.domain.folder.Folder
import com.xebialabs.xlrelease.domain.{Release, ScriptHelper}
import com.xebialabs.xlrelease.plugins.dashboard.cache.TileCacheConfiguration
import com.xebialabs.xlrelease.plugins.dashboard.cache.TileCacheConfiguration._
import com.xebialabs.xlrelease.plugins.dashboard.domain.Tile
import com.xebialabs.xlrelease.plugins.dashboard.service.TileScriptExecutor._
import com.xebialabs.xlrelease.script.builder.{ScriptContextBuilder, _}
import com.xebialabs.xlrelease.script.jython.{JythonScriptService, XlrJythonSupport}
import com.xebialabs.xlrelease.script.{EncryptionHelper, XlrScript, XlrScriptContext}
import grizzled.slf4j.Logging
import io.micrometer.core.annotation.Timed
import org.springframework.cache.annotation.{CacheConfig, CacheEvict, Cacheable}
import org.springframework.context.ApplicationContext
import org.springframework.stereotype.Component

import javax.script.ScriptContext
import scala.jdk.CollectionConverters._

@Component
@CacheConfig(cacheManager = TILE_CACHE_MANAGER, cacheResolver = TILE_CACHE_RESOLVER, keyGenerator = TILE_CACHE_KEY_GENERATOR,
  cacheNames = Array(TileCacheConfiguration.TILE_CACHE))
class TileScriptExecutor(val scriptService: JythonScriptService, applicationContext: ApplicationContext) extends Logging with XlrJythonSupport {

  @CacheEvict(condition = "#resolvedTile.cacheEnabled")
  @Timed
  def evictFromCache(parent: Option[ConfigurationItem], resolvedTile: Tile, additionalVariables: Map[String, Any]): Unit = {
    logger.debug(s"Executing refresh for a tile ${resolvedTile.getId}")
  }

  @Cacheable(condition = "#resolvedTile.cacheEnabled")
  @Timed
  @throws(classOf[Exception])
  def executeTileScript(parent: Option[ConfigurationItem], resolvedTile: Tile, additionalVariables: Map[String, Any]): AnyRef = {
    logger.debug(s"Executing tile script for a tile ${resolvedTile.getId}")
    val scriptContext = createScriptContext(parent, resolvedTile, additionalVariables, applicationContext)
    executeScript(scriptContext)
    scriptContext.getAttribute("data")
  }

}

private class TileScriptContextBuilder(parent: Option[ConfigurationItem],
                                       resolvedTile: Tile,
                                       additionalVariables: Map[String, Any],
                                       applicationContext: ApplicationContext) extends ScriptContextBuilder {
  withLogger().withScriptApi()
  withPythonSugar().withPythonGlobals().withPythonReleaseApi().withPythonUtilities()

  override protected def doBuild(context: XlrScriptContext): Unit = {
    EncryptionHelper.decrypt(resolvedTile)
    context.addProperties(resolvedTile, resolvedTile.getProperties)
    val properties: Map[String, Any] = additionalVariables ++ (parent match {
      case Some(release: Release) => Map("release" -> release)
      case Some(folder: Folder) => Map("folder" -> folder)
      case _ => Map.empty
    }) ++ Map("applicationContext" -> applicationContext, "tile" -> resolvedTile, "params" -> additionalVariables.asJava)
    properties.foreach { case (k, v) =>
      context.setAttribute(k, v, ScriptContext.ENGINE_SCOPE)
    }

    val script = ScriptHelper.getScript(resolvedTile)
    val name = s"${resolvedTile.getType}[${resolvedTile.getTitle}]"
    context.addScript(XlrScript.byContent(name = name, content = script, wrap = false, checkPermissions = false))
  }
}

private object TileScriptExecutor {
  def createScriptContext(parent: Option[ConfigurationItem], resolvedTile: Tile, additionalVariables: Map[String, Any], applicationContext: ApplicationContext): XlrScriptContext = {
    new TileScriptContextBuilder(parent, resolvedTile, additionalVariables, applicationContext).build()
  }
}
