import React from 'react';
import { DotEmptyState } from '@digital-ai/dot-components';
import { DashboardListPortal } from './dashboard-list.portal';
import { DashboardCards } from './dashboard-cards.component';
import { Dashboard } from '../../../types';
import { DashboardCardsSkeleton } from './dashboard-cards.skeleton';
import { Folder } from '../../../../../../../../../../../../core/xlr-ui/app/types';
import './dashboard-list.component.less';

export interface DashboardListProps {
    dashboardTitleFilter?: string;
    dashboards: Dashboard[];
    folder?: Folder;
    hasCreateDashboardPermission: boolean;
    isLoading: boolean;
    onCreate: () => void;
    onDelete: (dashboardId: string) => void;
    onDuplicate: (dashboard: Dashboard) => void;
    onFilterChange?: (title: string) => void;
}

export const DashboardList = ({
    dashboardTitleFilter = '',
    dashboards,
    folder,
    hasCreateDashboardPermission,
    isLoading,
    onCreate,
    onDelete,
    onDuplicate,
    onFilterChange,
}: DashboardListProps) => {
    const isButtonVisibleOnPortal = hasCreateDashboardPermission && (dashboards.length > 0 || !!dashboardTitleFilter);

    const handleDashboardFilterChange = (title: string) => {
        onFilterChange?.(title);
    };

    const handleDuplicateDashboard = (dashboard: Dashboard) => {
        onDuplicate(dashboard);
    };

    const handleDeleteDashboard = (dashboardId: string) => {
        onDelete(dashboardId);
    };

    const renderDashboardList = () => {
        if (dashboards.length || dashboardTitleFilter) {
            return (
                <DashboardCards
                    dashboardFilter={dashboardTitleFilter}
                    dashboards={dashboards}
                    folder={folder}
                    hasCreateDashboardPermission={hasCreateDashboardPermission}
                    onDashboardFilterChange={handleDashboardFilterChange}
                    onDelete={handleDeleteDashboard}
                    onDuplicate={handleDuplicateDashboard}
                />
            );
        } else {
            const subtitle = hasCreateDashboardPermission
                ? 'Get started by creating your first dashboard using pre-defined tiles that can be customized to meet your needs.'
                : 'You do not have the permissions required to create dashboards. Contact the Release administrator for assistance.';
            return (
                <DotEmptyState
                    buttonProps={hasCreateDashboardPermission ? { children: 'Create custom dashboard', onClick: onCreate } : undefined}
                    className="dashboard-empty-state"
                    imageSrc="static/24.1.20-622.113/include/DashboardPage/assets/data-analysis.svg"
                    subtitle={subtitle}
                    title="Custom dashboards"
                />
            );
        }
    };

    return (
        <>
            <DashboardListPortal isButtonVisible={isButtonVisibleOnPortal} onButtonClick={onCreate} />
            {isLoading && <DashboardCardsSkeleton />}
            {!isLoading && renderDashboardList()}
        </>
    );
};
