import { call, put } from 'redux-saga/effects';
import { SagaIterator } from 'redux-saga';
import { FolderDashboardSidebarProvider } from './xlr-folder-dashboard-provider';
import getAngularService from '../../../../../../../../../core/xlr-ui/app/features/common/services/angular-accessor';
import { navigation } from '../../../../../../../../../core/xlr-ui/app/features/main-navigation/ducks/navigation.reducer';
import { BreadcrumbItem, FolderPathService, MainNavigationItem } from '../../../../../../../../../core/xlr-ui/app/features/tasks/types/angular';
import { getFolderOrWaitForIt } from '../../../../../../../../../core/xlr-ui/app/features/folders/ducks/folders.saga';

const { updateBreadcrumbs, setSidebarMenu, setLastBreadcrumb, setBackItem } = navigation.actions;

describe('FolderDashboardSidebarProvider', () => {
    const dashboardsListPath = '/folders/Applications/Foldera23/Folderdc1/dashboards';
    const dashboardPath = `${dashboardsListPath}/Applications/Dashboard123`;
    const folderDashboardSidebarProvider = new FolderDashboardSidebarProvider();

    describe('isSupported', () => {
        it('should be false for not supported path', () => {
            expect(folderDashboardSidebarProvider.isSupported('/folders/')).toBe(false);
        });

        it('should return true for paths that start with /dashboards/Applications/Dashboard', () => {
            expect(folderDashboardSidebarProvider.isSupported('/folders/123/dashboards/Applications/Dashboard123')).toBe(true);
        });
    });

    describe('getDashboardsListPath', () => {
        it('should return correct path', () => {
            const path = '/folders/Applications/Foldera23/Folderdc1/dashboards/Applications/Dashboard123/Tileeaabb/details';
            expect(folderDashboardSidebarProvider.getDashboardsListPath(path)).toBe('/folders/Applications/Foldera23/Folderdc1/dashboards');
        });
    });

    describe('getDashboardPath', () => {
        it('should return correct path for non-details path', () => {
            expect(folderDashboardSidebarProvider.getDashboardPath(dashboardPath, false)).toBe(dashboardPath);
        });

        it('should return correct path for details path', () => {
            expect(folderDashboardSidebarProvider.getDashboardPath(`${dashboardPath}/Tileeaabb/details`, true)).toBe(dashboardPath);
        });
    });

    describe('getDashboardId', () => {
        const dashboardId = 'Applications/Dashboard123';

        it('should return correct ID when on dashboard page', () => {
            expect(folderDashboardSidebarProvider.getDashboardId(dashboardPath, false)).toBe(dashboardId);
        });

        it('should return correct ID when on dashboard details page', () => {
            expect(folderDashboardSidebarProvider.getDashboardId(`${dashboardPath}/Tileeaabb/details`, true)).toBe(dashboardId);
        });
    });

    describe('provide', () => {
        const xlrDashboardService = {
            loadConfiguration: jest.fn(),
        };

        const UiExtensionsService = {
            getMenuItems: jest.fn(),
        };

        const folderPathService: FolderPathService = {
            getPathFromParentFolderId: () => [
                {
                    id: 'Applications',
                    title: 'My Root',
                },
                {
                    id: 'Applications/Foldera23',
                    title: 'My Folder',
                },
            ],
            getPathFromReleaseId: jest.fn(),
        };

        const $state = {
            params: {
                page: 'dashboards',
            },
        };

        const dashboard = {
            title: 'my dashboard',
            id: 'Applications/Dashboard077f84862b9e4849a867fde300184e91',
        };

        const selectedFolder = {
            id: 'Applications/Foldera23/Folderdc1',
            title: 'My Folder',
        };

        const path = `/folders/${selectedFolder.id}/dashboards/${dashboard.id}`;

        const folderTabs: MainNavigationItem[] = [
            {
                icon: 'dashboard',
                label: 'Custom dashboards',
                pathSuffix: 'dashboards',
                weight: 30,
                permitted: true,
            },
        ];

        const breadcrumbs: BreadcrumbItem[] = [
            {
                href: '/folders/Applications/Foldera23/dashboards',
                text: 'Folder 1',
                underline: 'hover',
            },
            {
                href: '/folders/Applications/Foldera23/Folderdc1/dashboards',
                text: 'My Folder',
                underline: 'hover',
            },
            {
                href: '/folders/Applications/Foldera23/Folderdc1/dashboards',
                text: 'Custom dashboards',
                underline: 'hover',
            },
            {
                href: '/folders/Applications/Foldera23/Folderdc1/dashboards/Applications/Dashboard077f84862b9e4849a867fde300184e91',
                text: 'my dashboard',
                underline: 'hover',
            },
        ];

        describe('provide', () => {
            const verifyCommonProvideSagas = (gen: SagaIterator) => {
                expect(gen.next().value).toMatchObject(call(getAngularService, 'xlrelease.XlrDashboardService'));
                expect(gen.next(xlrDashboardService).value).toMatchObject(call([xlrDashboardService, xlrDashboardService.loadConfiguration], dashboard.id));

                expect(gen.next({ data: dashboard }).value).toMatchObject(call(getFolderOrWaitForIt));
                expect(gen.next(selectedFolder).value).toMatchObject(call(getAngularService, 'UiExtensionsService'));
                expect(gen.next(UiExtensionsService).value).toMatchObject(
                    call([UiExtensionsService, UiExtensionsService.getMenuItems], 'xlrelease.sidebar.Folder', selectedFolder),
                );
                expect(gen.next(folderTabs).value).toMatchObject(call(getAngularService, '$state'));
                expect(gen.next($state).value).toMatchObject(
                    put(
                        setSidebarMenu({
                            sidebarMenu: [
                                {
                                    pathSuffix: `folders/${selectedFolder.id}/dashboards`,
                                    selected: true,
                                    startIconId: 'dashboard',
                                    text: 'Custom dashboards',
                                },
                            ],
                            withFolder: true,
                        }),
                    ),
                );

                expect(gen.next().value).toMatchObject(call(getAngularService, 'FolderPathService'));
                expect(gen.next(folderPathService).value).toMatchObject(
                    call([folderPathService, folderPathService.getPathFromParentFolderId], selectedFolder.id),
                );
                expect(
                    gen.next([
                        {
                            id: 'Applications/Foldera23',
                            title: 'Folder 1',
                        },
                        {
                            id: selectedFolder.id,
                            title: selectedFolder.title,
                        },
                    ]).value,
                ).toMatchObject(put(setLastBreadcrumb(undefined)));
                expect(gen.next().value).toMatchObject(put(setBackItem(null)));
            };

            it('should call proper sagas when NOT in dashboard details', () => {
                const gen: SagaIterator = folderDashboardSidebarProvider.provide(path);
                verifyCommonProvideSagas(gen);
                expect(gen.next().value).toMatchObject(put(updateBreadcrumbs(breadcrumbs)));
                expect(gen.next().done).toBe(true);
            });

            it('should call proper sagas when in dashboard details', () => {
                const gen: SagaIterator = folderDashboardSidebarProvider.provide(`${path}/Tileabc/details`);
                verifyCommonProvideSagas(gen);
                expect(gen.next().value).toMatchObject(put(updateBreadcrumbs([...breadcrumbs, { text: 'Details' }])));
                expect(gen.next().done).toBe(true);
            });
        });
    });
});
