import React, { ChangeEvent, MouseEvent, useState } from 'react';
import { DotCard, DotEmptyState, DotIcon, DotIconButton, DotInputText, DotLink, DotTypography } from '@digital-ai/dot-components';
import { Dashboard } from '../../../types';
import { checkIfDashboardHasEditPermission } from '../../../helper';
import { DashboardActionContextMenu } from '../dashboard-action-context-menu.component';
import { Folder } from '../../../../../../../../../../../../core/xlr-ui/app/types';
import { AuthenticatorService } from '../../../../../../../../../../../../core/xlr-ui/app/features/tasks/types/angular';
import getAngularService from '../../../../../../../../../../../../core/xlr-ui/app/features/common/services/angular-accessor';
import './dashboard-cards.component.less';

export interface DashboardCardsProps {
    dashboardFilter?: string;
    dashboards: Dashboard[];
    folder?: Folder;
    hasCreateDashboardPermission: boolean;
    onDashboardFilterChange: (title: string) => void;
    onDelete: (dashboardId: string) => void;
    onDuplicate: (dashboard: Dashboard) => void;
}

export const DashboardCards = ({
    dashboardFilter,
    dashboards,
    folder,
    hasCreateDashboardPermission,
    onDashboardFilterChange,
    onDelete,
    onDuplicate,
}: DashboardCardsProps) => {
    const [contextMenuAnchorEl, setContextMenuAnchorEl] = useState<HTMLElement | null>(null);
    const [contextMenuDashboard, setContextMenuDashboard] = useState<Dashboard | null>(null);
    const authenticator = getAngularService('Authenticator') as unknown as AuthenticatorService;
    const hasDeletePermission = contextMenuDashboard ? checkIfDashboardHasEditPermission(authenticator, contextMenuDashboard) : false;
    const baseDashboardPath = folder ? `#/folders/${folder.id}` : '#';

    const toggleContextMenu = (event: MouseEvent<HTMLElement>, dashboard: Dashboard) => {
        const isSameDashboard = dashboard.id === contextMenuDashboard?.id;
        setContextMenuAnchorEl(event.currentTarget);
        setContextMenuDashboard(isSameDashboard ? null : dashboard);
    };
    const handleDashboardFilterChange = (event: ChangeEvent<HTMLInputElement | HTMLTextAreaElement>) => {
        onDashboardFilterChange(event.target.value);
    };

    return (
        <div className="dashboard-cards">
            <div className="dashboard-cards-header">
                <DotTypography data-testid="page-title" variant="h1">
                    Custom dashboards
                </DotTypography>
                <DotInputText
                    autoFocus={true}
                    fullWidth={false}
                    hasDebounce={true}
                    id="dashboard-search"
                    name="dashboard-search"
                    onChange={handleDashboardFilterChange}
                    placeholder="Filter by name"
                    startIcon={<DotIcon iconId="search" />}
                    value={dashboardFilter}
                />
            </div>
            <div className="dashboard-cards-list">
                {dashboards.length ? (
                    dashboards.map((dashboard) => {
                        const hasEditPermission = checkIfDashboardHasEditPermission(authenticator, dashboard);
                        const hasContextMenuIconButton = hasCreateDashboardPermission || hasEditPermission;
                        return (
                            <DotCard className="dashboard-card" key={dashboard.id}>
                                <div className="dashboard-card-row">
                                    <div className="dashboard-card-row-left">
                                        <DotIcon data-testid="card-icon" iconId="dashboard" />
                                        <div>
                                            <DotLink
                                                ariaLabel={dashboard.title}
                                                data-testid="dashboard-link"
                                                href={`${baseDashboardPath}/dashboards/${dashboard.id}`}
                                                underline="none"
                                            >
                                                <DotTypography variant="subtitle2">{dashboard.title}</DotTypography>
                                            </DotLink>
                                            {folder && (
                                                <DotLink
                                                    ariaLabel={folder.title}
                                                    data-testid="folder-link"
                                                    href={`${baseDashboardPath}/templates`}
                                                    underline="none"
                                                >
                                                    <DotTypography variant="body1">{folder.title}</DotTypography>
                                                </DotLink>
                                            )}
                                        </div>
                                    </div>
                                    {hasContextMenuIconButton && (
                                        <DotIconButton
                                            data-testid="task-options-button"
                                            iconId="options"
                                            onClick={(e) => {
                                                e.stopPropagation();
                                                toggleContextMenu(e, dashboard);
                                            }}
                                        />
                                    )}
                                </div>
                            </DotCard>
                        );
                    })
                ) : (
                    <DotEmptyState
                        className="dashboard-cards-empty-state"
                        imageSrc="static/@project.version@/include/DashboardPage/assets/data-analysis.svg"
                        subtitle="Please adjust the filters and attempt your search again"
                        title="No results match the current filter selection"
                    />
                )}
            </div>

            <DashboardActionContextMenu
                anchorEl={contextMenuAnchorEl || undefined}
                dashboard={contextMenuDashboard}
                hasCreateDashboardPermission={hasCreateDashboardPermission}
                hasDeleteDashboardPermission={hasDeletePermission}
                onDelete={onDelete}
                onDuplicate={onDuplicate}
                onLeave={() => setContextMenuDashboard(null)}
            />
        </div>
    );
};
