import React from 'react';
import { mountWithTheme, ReactWrapper } from '../../../../../../../../../../../../core/xlr-ui/tests/unit/testing-utils';
import { DashboardList, DashboardListProps } from './dashboard-list.component';
import { dashboardsMock } from '../../../mocks';
import { DashboardListPortal } from './dashboard-list.portal';
import { DashboardCards } from './dashboard-cards.component';
import { DashboardCardsSkeleton } from './dashboard-cards.skeleton';
import { DotEmptyState } from '@digital-ai/dot-components';
import { folderMock, foldersMock } from '../../../../../../../../../../../../core/xlr-ui/app/features/tasks/__mocks__/release.mock';
import * as angularAccessor from '../../../../../../../../../../../../core/xlr-ui/app/features/common/services/angular-accessor';

describe('DashboardList', () => {
    let wrapper: ReactWrapper;

    const dashboards = dashboardsMock;
    const hasCreateDashboardPermission = true;
    const isLoading = false;
    const onCreate = jest.fn();
    const onDelete = jest.fn();
    const onDuplicate = jest.fn();
    const onFilterChange = jest.fn();

    const defaultProps: DashboardListProps = {
        dashboards,
        hasCreateDashboardPermission,
        isLoading,
        onCreate,
        onDelete,
        onDuplicate,
    };

    const mount = (props = defaultProps) => {
        wrapper = mountWithTheme(<DashboardList {...props} />);
    };

    const getDashboardListPortal = () => wrapper.find(DashboardListPortal);
    const getDashboardCards = () => wrapper.find(DashboardCards);
    const getDotEmptyState = () => wrapper.find(DotEmptyState);
    const getGlobalDashboardCardsSkeleton = () => wrapper.find(DashboardCardsSkeleton);

    beforeEach(() => {
        const authenticator = {
            hasPermission: jest.fn(),
        };
        const getAngularServiceSpy = jest.spyOn(angularAccessor, 'default') as unknown as jest.SpyInstance;
        getAngularServiceSpy.mockReturnValue(authenticator);
    });

    afterEach(() => jest.resetAllMocks());

    it('should render correct elements when at least one dashboard is defined', () => {
        mount();
        const dashboardListPortal = getDashboardListPortal();
        expect(dashboardListPortal).toExist();
        expect(dashboardListPortal.props().isButtonVisible).toBe(true);

        const dashboardCards = getDashboardCards();
        expect(dashboardCards).toExist();
        const dashboardCardsProps = dashboardCards.props();
        expect(dashboardCardsProps.dashboardFilter).toBe('');
        expect(dashboardCardsProps.dashboards).toStrictEqual(dashboards);
        expect(dashboardCardsProps.folder).toBeUndefined();

        const globalDashboardCardsSkeleton = getGlobalDashboardCardsSkeleton();
        expect(globalDashboardCardsSkeleton).not.toExist();
    });

    it('should render dashboard cards with proper folder and dashboard filter', () => {
        const titleFilter = 'my title';
        const folder = folderMock;
        mount({ ...defaultProps, dashboardTitleFilter: titleFilter, folder });
        const dashboardCards = getDashboardCards();
        expect(dashboardCards).toExist();
        const dashboardCardsProps = dashboardCards.props();
        expect(dashboardCardsProps.dashboardFilter).toBe(titleFilter);
        expect(dashboardCardsProps.folder).toStrictEqual(folder);
    });

    it('should render empty state component when no dashboards and no filters', () => {
        mount({ ...defaultProps, dashboards: [], dashboardTitleFilter: '' });
        const dashboardCards = getDashboardCards();
        expect(dashboardCards).not.toExist();

        const dotEmptyState = getDotEmptyState();
        expect(dotEmptyState).toExist();
        const dotEmptyStateProps = dotEmptyState.props();
        expect(dotEmptyStateProps.className).toBe('dashboard-empty-state');
        expect(dotEmptyStateProps.imageSrc).toBe('static/24.1.0-1228.113/include/DashboardPage/assets/data-analysis.svg');
        expect(dotEmptyStateProps.subtitle).toBe(
            'Get started by creating your first dashboard using pre-defined tiles that can be customized to meet your needs.',
        );
        expect(dotEmptyStateProps.title).toBe('Custom dashboards');
    });

    it('should render empty state component when no dashboards and filters', () => {
        mount({ ...defaultProps, dashboards: [], dashboardTitleFilter: 'my filter' });

        const dotEmptyState = getDotEmptyState();
        expect(dotEmptyState).toExist();
        const dotEmptyStateProps = dotEmptyState.props();
        expect(dotEmptyStateProps.className).toBe('dashboard-cards-empty-state');
        expect(dotEmptyStateProps.imageSrc).toBe('static/24.1.0-1228.113/include/DashboardPage/assets/data-analysis.svg');
        expect(dotEmptyStateProps.subtitle).toBe("We couldn't find any data that matches your filter criteria");
        expect(dotEmptyStateProps.title).toBe('No results found');
    });

    it('should render empty state with correct subtitle when no dashboards and user does not have a permission to create dashboards', () => {
        mount({ ...defaultProps, dashboards: [], hasCreateDashboardPermission: false });
        const dotEmptyState = getDotEmptyState();
        expect(dotEmptyState).toExist();
        expect(dotEmptyState.props().subtitle).toBe(
            'You do not have the permissions required to create dashboards. Contact the Release administrator for assistance.',
        );
    });

    it('should dispatch correct action when onDelete event has been triggered', () => {
        const titleFilter = 'my title';
        mount({ ...defaultProps, dashboardTitleFilter: titleFilter });
        const dashboard = dashboards[0];
        const dashboardCards = getDashboardCards();
        dashboardCards.invoke('onDelete')(dashboard.id as string);
        expect(onDelete).toHaveBeenCalledWith(dashboard.id);
    });

    it('should dispatch correct action when onDuplicate event has been triggered', () => {
        const titleFilter = 'my title';
        mount({ ...defaultProps, dashboardTitleFilter: titleFilter });
        const dashboard = dashboards[0];
        const dashboardCards = getDashboardCards();
        dashboardCards.invoke('onDuplicate')(dashboard);
        expect(onDuplicate).toHaveBeenCalledWith(dashboard);
    });

    it('should render correct dashboards and execute correct handler when filter is set with onFilterChange defined', () => {
        const filter = 'Dashboard 1';
        mount({ ...defaultProps, onFilterChange });
        const dashboardCards = getDashboardCards();
        dashboardCards.invoke('onDashboardFilterChange')(filter);
        expect(onFilterChange).toHaveBeenCalledWith(filter);
    });

    it('should handle onButtonClick event on the DashboardListPortal component', () => {
        mount();
        const dashboardListPortal = getDashboardListPortal();
        dashboardListPortal.invoke('onButtonClick')();
        expect(onCreate).toHaveBeenCalled();
    });

    it('should render skeletons when loading', () => {
        mount({ ...defaultProps, isLoading: true });
        const globalDashboardCardsSkeleton = getGlobalDashboardCardsSkeleton();
        expect(globalDashboardCardsSkeleton).toExist();
    });

    it('should render DashboardCards with folderId prop when folderId is defined', () => {
        const folder = foldersMock[0];
        mount({ ...defaultProps, folder });

        const dashboardCards = getDashboardCards();
        expect(dashboardCards.props().folder).toStrictEqual(folder);
    });
});
