package com.xebialabs.xlrelease.plugins.dashboard.service

import com.codahale.metrics.annotation.Timed
import com.github.benmanes.caffeine.cache.Cache
import com.xebialabs.deployit.booter.local.utils.Strings.isNotBlank
import com.xebialabs.deployit.checks.Checks
import com.xebialabs.deployit.checks.Checks.checkArgument
import com.xebialabs.deployit.plugin.api.reflect.{Descriptor, DescriptorRegistry, Type}
import com.xebialabs.deployit.security.Permissions.{authenticationToPrincipals, getAuthenticatedUserName, getAuthentication}
import com.xebialabs.deployit.security.{PermissionDeniedException, RoleService}
import com.xebialabs.xlrelease.api.internal.DecoratorsCache
import com.xebialabs.xlrelease.domain.ScriptHelper.{getScript, readScript}
import com.xebialabs.xlrelease.events.XLReleaseEventBus
import com.xebialabs.xlrelease.plugins.dashboard.cache.TileCacheConfiguration.TILE_CACHE_MANAGER
import com.xebialabs.xlrelease.plugins.dashboard.domain.{Dashboard, Tile, TileScope}
import com.xebialabs.xlrelease.plugins.dashboard.events.{DashboardCreatedEvent, DashboardDeletedEvent, DashboardUpdatedEvent}
import com.xebialabs.xlrelease.plugins.dashboard.repository.DelegatingDashboardRepository
import com.xebialabs.xlrelease.repository.Ids.isNullId
import com.xebialabs.xlrelease.script.builder.ScriptContextBuilder
import com.xebialabs.xlrelease.script.jython.{JythonScriptService, XlrJythonSupport}
import com.xebialabs.xlrelease.script.{XlrScript, XlrScriptContext}
import com.xebialabs.xlrelease.serialization.json.utils.CiSerializerHelper.deserialize
import com.xebialabs.xlrelease.service.CiIdService
import com.xebialabs.xlrelease.utils.PasswordVerificationUtils._
import grizzled.slf4j.Logging
import org.springframework.beans.factory.annotation.Qualifier
import org.springframework.cache.CacheManager
import org.springframework.cache.interceptor.SimpleKey
import org.springframework.stereotype.Service

import javax.script.ScriptContext
import scala.jdk.CollectionConverters._

@Service
class DashboardService(dashboardRepository: DelegatingDashboardRepository,
                       dashboardSecurity: DashboardSecurity,
                       dashboardSecurityDecorator: DashboardSecurityDecorator,
                       @Qualifier(TILE_CACHE_MANAGER)
                       cacheManager: CacheManager,
                       roleService: RoleService,
                       val scriptService: JythonScriptService,
                       val eventBus: XLReleaseEventBus,
                       implicit val ciIdService: CiIdService) extends Logging with XlrJythonSupport {
  @Timed
  def exists(dashboardId: String): Boolean = dashboardRepository.exists(dashboardId)

  @Timed
  def search(parentId: String, title: String = null, enforcePermissions: Boolean = true): Seq[Dashboard] =
    decorateWithEffectiveSecurity(dashboardRepository.search(parentId, title, currentPrincipals, currentRoles, enforcePermissions))

  @Timed
  def findDashboardById(dashboardId: String): Dashboard = {
    val dashboard = dashboardRepository.findDashboardById(dashboardId)
    decorateWithEffectiveSecurity(Seq(dashboard))
    dashboard
  }

  @Timed
  def createDashboard(dashboard: Dashboard): Dashboard = {
    validate(dashboard)
    setOwnerIfMissing(dashboard)
    provisionDashboard(dashboard)
    val createdDashboard = dashboardRepository.createDashboard(dashboard)
    eventBus.publish(DashboardCreatedEvent(createdDashboard))
    dashboardSecurity.savePermissions(createdDashboard)
    createdDashboard
  }

  @Timed
  def updateDashboard(dashboard: Dashboard): Dashboard = {
    validate(dashboard)
    setOwnerIfMissing(dashboard)
    provisionDashboardTiles(dashboard)
    val originalDashboard = dashboardRepository.findDashboardById(dashboard.getId)
    replacePasswordPropertiesInCiIfNeeded(Some(originalDashboard), dashboard)
    val updatedDashboard = dashboardRepository.updateDashboard(dashboard)
    eventBus.publish(DashboardUpdatedEvent(updatedDashboard))
    dashboardSecurity.savePermissions(updatedDashboard)
    updatedDashboard
  }

  @Timed
  def deleteDashboard(dashboardId: String): Unit = {
    if (dashboardId.equals(Dashboard.HOME_DASHBOARD_ID)) {
      throw new PermissionDeniedException("Home dashboard can't be deleted")
    }
    dashboardSecurity.clearPermissions(dashboardId)
    val dashboard = dashboardRepository.findDashboardById(dashboardId)
    dashboardRepository.deleteDashboard(dashboardId)
    eventBus.publish(DashboardDeletedEvent(dashboard))
  }

  @Timed
  def getDashboardTemplates(scope: TileScope): Seq[Descriptor] = {
    DescriptorRegistry.getSubtypes(Type.valueOf(classOf[Dashboard]))
      .asScala
      .map(_.getDescriptor)
      .filter(desc => !desc.isVirtual && desc.newInstance[Dashboard]("dummy").isSupportedOn(scope))
      .toList
      .sortBy(_.getLabel)
  }

  @Timed
  def evictTilesCache(dashboard: Dashboard): Unit = {
    cacheManager.getCacheNames.asScala
      .foreach(cacheName => {
        val cache = cacheManager.getCache(cacheName).getNativeCache.asInstanceOf[Cache[SimpleKey, Any]]
        cache.asMap().asScala.keys
          .filter(key => key.toString.contains(dashboard.getId))
          .foreach(key => {
            cacheManager.getCache(cacheName).evict(key)
          })
      })
  }

  def provisionDashboard(dashboard: Dashboard): Unit = {
    if (isNotBlank(dashboard.getTemplateLocation)) {
      val template = deserialize(readScript(dashboard.getTemplateLocation), null).asInstanceOf[Dashboard]
      dashboard.setTiles(template.getTiles)
    }
    if (isNotBlank(dashboard.getScriptLocation)) {
      executeProvisionScript(dashboard)
    }
  }

  def provisionDashboardTiles(dashboard: Dashboard): Unit = {
    dashboard.setTiles(dashboard.getTiles.asScala.map(executeTileProvisionScript(dashboard, _)).asJava)
  }

  protected def decorateWithEffectiveSecurity(dashboards: Seq[Dashboard]): Seq[Dashboard] = {
    dashboardSecurityDecorator.decorate(dashboards.asJava, new DecoratorsCache().getDecoratorCache[DashboardSecurityDecoratorCache](""))
    dashboards
  }

  private def validate(dashboard: Dashboard): Unit = {
    if (!dashboard.isReleaseDashboard && !dashboard.isHomeDashboard) { // don't ask
      checkArgument(isNotBlank(dashboard.getTitle), "Title must be set")
      dashboardSecurity.validate(dashboard)
    }

    if (dashboard.getAutoRefresh && (dashboard.getAutoRefreshInterval == null || dashboard.getAutoRefreshInterval < 1)) {
      throw new Checks.IncorrectArgumentException("Auto refresh interval must be larger then 0", dashboard.getAutoRefreshInterval)
    }
  }

  private def setOwnerIfMissing(dashboard: Dashboard): Unit = {
    if (!dashboard.hasOwner) {
      dashboard.setOwner(getAuthenticatedUserName)
    }
  }

  private def currentPrincipals = authenticationToPrincipals(getAuthentication).asScala

  private def currentRoles = roleService.getRolesFor(getAuthentication).asScala

  private def executeProvisionScript(dashboard: Dashboard): Dashboard = if (dashboard.hasConfigurationScript) {
    val scriptContext = DashboardScriptContext(dashboard)
    executeScript(scriptContext)
    scriptContext.getAttribute("dashboard").asInstanceOf[Dashboard]
  } else {
    dashboard
  }


  private def executeTileProvisionScript(dashboard: Dashboard, tile: Tile): Tile = {
    val scriptLocation = tile.getProvisioningScriptLocation
    if (isNullId(tile.getId) && isNotBlank(scriptLocation)) {
      val scriptContext = DashboardScriptContext(dashboard, tile)
      executeScript(scriptContext)
      scriptContext.getAttribute("tile").asInstanceOf[Tile]
    } else {
      tile
    }
  }

}

private class DashboardTileScriptContextBuilder(dashboard: Dashboard, tile: Tile) extends ScriptContextBuilder {
  withLogger().withScriptApi()
  withPythonSugar().withPythonGlobals().withPythonReleaseApi().withPythonUtilities()

  override protected def doBuild(context: XlrScriptContext): Unit = {
    context.setAttribute("dashboard", dashboard, ScriptContext.ENGINE_SCOPE)
    context.setAttribute("tile", tile, ScriptContext.ENGINE_SCOPE)

    val scriptLocation = tile.getProvisioningScriptLocation
    context.addScript(XlrScript.byResource(resource = scriptLocation, wrap = false, checkPermissions = false))
  }

}


private class DashboardScriptContextBuilder(dashboard: Dashboard) extends ScriptContextBuilder {
  withLogger().withScriptApi()
  withPythonSugar().withPythonGlobals().withPythonReleaseApi().withPythonUtilities()

  override protected def doBuild(context: XlrScriptContext): Unit = {
    context.setAttribute("dashboard", dashboard, ScriptContext.ENGINE_SCOPE)

    val xlrScriptName = s"${dashboard.getType.toString}[${dashboard.getId}]"
    context.addScript(XlrScript.byContent(name = xlrScriptName, content = getScript(dashboard), wrap = false, checkPermissions = false))
  }

}

object DashboardScriptContext {
  def apply(dashboard: Dashboard): XlrScriptContext = new DashboardScriptContextBuilder(dashboard).build()

  def apply(dashboard: Dashboard, tile: Tile): XlrScriptContext = new DashboardTileScriptContextBuilder(dashboard, tile).build()
}
