import React from 'react';
import moment from 'moment';
import { DotAvatar, DotTable, DotTypography } from '@digital-ai/dot-components';
import { mockResizeObserver, mountWithStoreAndTheme, ReactWrapper } from '../../../../../../../../../../core/xlr-ui/tests/unit/testing-utils';
import { primaryButton, secondaryButton, tertiaryButton } from '../../CommonCardTile/tile-buttons-row.mock';
import { CommonActiveTile } from '../../CommonCardTile/common-active-tile.component';
import { ActiveWorkflowTile, ActiveWorkflowTileProps } from './active-workflow-tile.component';
import { WorkflowTileToggleEnum } from './workflow-tile.component';
import { HomeWorkflow, workflow as workflowReducer } from '../../../../../../../../../../core/xlr-ui/app/features/workflow/ducks/workflow.reducer';
import { DATE_FORMAT_DAY_FIRST } from '../../../../../../../../../../core/xlr-ui/app/js/locale/constants';
import * as angularAccessor from '../../../../../../../../../../core/xlr-ui/app/features/common/services/angular-accessor';
import { getReleaseIdForUrl } from '../../TemplateTile/components/helpers/helpers';
import { ChipGroup } from '../../../../../../../../../../core/xlr-ui/app/react/components/chip-group/chip-group.component';

const { init } = workflowReducer.actions;

describe('ActiveWorkflowTile', () => {
    const dispatch = jest.fn();

    const title = 'Workflow executions';
    const subtitle = 'Account workflows in last 30 days - top 5';
    const workflows: Array<HomeWorkflow> = [
        {
            numberOfExecutions: 2,
            categories: ['Application onboarding', 'Security'],
            releaseId: 'myId1',
            logoId: 'logoId',
            releaseTitle: 'My first workflow',
            folderId: 'folderId',
            folderName: 'folder1',
            folderPath: '/',
        },
        {
            numberOfExecutions: 1,
            categories: ['Application onboarding'],
            releaseEndDate: 1695168000000,
            releaseId: 'myId2',
            logoId: null,
            releaseTitle: 'My second workflow',
            releaseStartDate: 1694429074867,
            releaseStatus: 'in_progress',
            folderId: 'folderId',
            folderName: 'folder1',
            folderPath: '/',
        },
    ];
    const headerEndContent = <div>This is end content</div>;
    const tableType = WorkflowTileToggleEnum.MostPopular;

    const defaultProps: ActiveWorkflowTileProps = {
        headerEndContent,
        primaryButton,
        secondaryButton,
        subtitle,
        tableType,
        tertiaryButton,
        title,
        workflows,
    };

    const defaultState = {
        profile: { dateFormat: DATE_FORMAT_DAY_FIRST },
    };

    const mount = (props: ActiveWorkflowTileProps = defaultProps, state = defaultState) => {
        return mountWithStoreAndTheme(<ActiveWorkflowTile {...props} />, dispatch, state);
    };

    const getCommonActiveTile = (wrapper: ReactWrapper) => wrapper.find(CommonActiveTile);
    const getTable = (wrapper: ReactWrapper) => wrapper.find(DotTable);
    const getWorkflowNameData = (wrapper: ReactWrapper) =>
        wrapper.findWhere((node) => node.is(DotTypography) && node.props()['data-testid'] === 'row-workflow-title');
    const getWorkflowLogoData = (wrapper: ReactWrapper) => wrapper.find('.workflow-name').find(DotAvatar);
    const getCategoryChipGroup = (wrapper: ReactWrapper) => wrapper.find(ChipGroup);
    const getNumberOfExecutionsData = (wrapper: ReactWrapper) =>
        wrapper.findWhere((node) => node.is(DotTypography) && node.props()['data-testid'] === 'row-workflow-executions');

    beforeEach(() => {
        const getAngularServiceSpy = jest.spyOn(angularAccessor, 'default') as unknown as jest.SpyInstance;
        const dateFilterWrapper = (_filter: string) => (date: string, angularJsFormat: string) => moment(date).format(angularJsFormat);
        getAngularServiceSpy.mockReturnValue(dateFilterWrapper);
    });

    beforeAll(() => {
        mockResizeObserver();
    });

    afterAll(() => {
        jest.restoreAllMocks();
    });

    it('should render properly', () => {
        const wrapper = mount();

        const commonActiveTileProps = getCommonActiveTile(wrapper).props();
        expect(commonActiveTileProps.avatarIcon).toBe('workflow');
        expect(commonActiveTileProps.headerEndContent).toStrictEqual(headerEndContent);
        expect(commonActiveTileProps.primaryButton).toBe(primaryButton);
        expect(commonActiveTileProps.secondaryButton).toBe(secondaryButton);
        expect(commonActiveTileProps.subtitle).toBe(subtitle);
        expect(commonActiveTileProps.tertiaryButton).toBe(tertiaryButton);
        expect(commonActiveTileProps.title).toBe(title);
    });

    describe('main content', () => {
        it('should have correct table data', () => {
            const wrapper = mount();

            const workflowName = getWorkflowNameData(wrapper).at(0);
            expect(workflowName.props().variant).toBe('body1');
            expect(workflowName).toHaveText(workflows[0].releaseTitle);

            const workflowLogo1 = getWorkflowLogoData(wrapper).at(0);
            expect(workflowLogo1.props().alt).toBe(workflows[0].releaseTitle);
            expect(workflowLogo1.props().size).toBe('small');
            expect(workflowLogo1.props().type).toBe('image');
            expect(workflowLogo1.props().imageSrc).toBe(`api/v1/templates/logo/${workflows[0].logoId}`);

            const workflowLogo2 = getWorkflowLogoData(wrapper).at(1);
            expect(workflowLogo2.props().alt).toBe(workflows[1].releaseTitle);
            expect(workflowLogo2.props().size).toBe('small');
            expect(workflowLogo2.props().type).toBe('icon');
            expect(workflowLogo2.props().iconId).toBe('workflow');

            const categoryChipGroup = getCategoryChipGroup(wrapper);
            expect(categoryChipGroup).toHaveLength(workflows.length);
            expect(categoryChipGroup.at(0).props().labels).toBe(workflows[0].categories);
            expect(categoryChipGroup.at(1).props().labels).toBe(workflows[1].categories);

            expect(getNumberOfExecutionsData(wrapper).at(0)).toHaveText(`${workflows[0].numberOfExecutions}`);
        });

        describe('when "Most popular" is selected', () => {
            it('should render correctly', () => {
                const wrapper = mount();

                const columns = [
                    { id: 'workflowName', label: 'Workflow', truncate: true },
                    { id: 'numberOfExecutions', label: 'No. of executions', width: '152px' },
                    { id: 'category', label: 'Category', width: '37%' },
                ];

                const tableProps = getTable(wrapper).props();
                expect(tableProps.className).toBe('active-workflow-table');
                expect(tableProps.columns).toStrictEqual(columns);
                expect(tableProps.sortable).toBe(false);
            });

            it('should handle on click for table row', () => {
                const wrapper = mount();
                const table = getTable(wrapper);
                table.invoke('onRowClick')?.({} as never, workflows[0].releaseTitle);
                expect(dispatch).toBeCalledWith(init({ searchInput: workflows[0].releaseTitle }));
            });
        });

        describe('when "Recent" is selected', () => {
            it('should render correctly', () => {
                const wrapper = mount({ ...defaultProps, tableType: WorkflowTileToggleEnum.Recent });
                const columns = [
                    { id: 'workflowName', label: 'Workflow', truncate: true },
                    { id: 'status', label: 'Status', width: '136px' },
                    { id: 'ended', label: 'Ended', width: '136px' },
                    { id: 'category', label: 'Category', width: '37%' },
                ];
                expect(getTable(wrapper).props().columns).toStrictEqual(columns);
            });

            it('should handle on click for table row', () => {
                const wrapper = mount({ ...defaultProps, tableType: WorkflowTileToggleEnum.Recent });
                const table = getTable(wrapper);
                table.invoke('onRowClick')?.({} as never, getReleaseIdForUrl(workflows[0]));
                expect(window.location.href).toStrictEqual(`http://localhost/#/stepper/${getReleaseIdForUrl(workflows[0])}`);
            });
        });

        describe('when "Running" is selected', () => {
            it('should render correctly', () => {
                const wrapper = mount({ ...defaultProps, tableType: WorkflowTileToggleEnum.Running });
                const columns = [
                    { id: 'workflowName', label: 'Workflow', truncate: true },
                    { id: 'status', label: 'Status', width: '136px' },
                    { id: 'started', label: 'Started', width: '136px' },
                    { id: 'category', label: 'Category', width: '37%' },
                ];
                expect(getTable(wrapper).props().columns).toStrictEqual(columns);
            });
        });
    });
});
