import React from 'react';
import classNames from 'classnames';
import moment from 'moment';
import { DotIcon, DotPill, DotTable, DotTooltip, DotTypography } from '@digital-ai/dot-components';
import { ButtonWithLink } from '../../CommonCardTile/common-card-tile.component';
import { CommonActiveTile } from '../../CommonCardTile/common-active-tile.component';
import { getReleaseIdForUrl } from '../../TemplateTile/components/helpers/helpers';
import { calculateStatusColors, calculateStatusIcon, formatReleaseDuration, formatStatusText } from './helpers/helpers';
import { useAppSelector } from '../../../../../../../../../../core/xlr-ui/app/js/hooks';
import { getProfile } from '../../../../../../../../../../core/xlr-ui/app/features/profile/ducks/profile.selectors';
import {
    calculateDotDateFormat,
    getDaysHoursMinutesFromMS,
} from '../../../../../../../../../../core/xlr-ui/app/features/tasks/components/rails/scheduling/helper';
import { RegionalSettings } from '../../../../../../../../../../core/xlr-ui/app/js/locale/regional-settings';
import './active-release-tile.component.less';

export interface HomeRelease {
    folderId: string;
    folderName: string;
    folderPath: string;
    lastEditedByUser: number;
    releaseEndDate: number;
    releaseId: string;
    releaseStartDate: number;
    releaseStatus: string;
    releaseTitle: string;
}

export interface ActiveReleaseTileProps {
    primaryButton: ButtonWithLink;
    releases: Array<HomeRelease>;
    secondaryButton?: ButtonWithLink;
    title: string;
}

export const ActiveReleaseTile = ({ primaryButton, releases, secondaryButton, title }: ActiveReleaseTileProps) => {
    const { dateFormat: profileDateFormat } = useAppSelector(getProfile);
    const dateFormat = calculateDotDateFormat(RegionalSettings.getDateFormats(profileDateFormat).shortDate);

    const tableColumns = [
        { id: 'releases', label: 'Release', truncate: true },
        { id: 'status', label: 'Status', width: '136px' },
        { id: 'endDate', label: 'End date', width: '136px' },
        { id: 'duration', label: 'Duration', width: '136px' },
    ];

    const releasesData = releases.map((release) => {
        const releaseId = getReleaseIdForUrl(release);
        const { background: statusBackground, border: statusBorder } = calculateStatusColors(release.releaseStatus);
        const [durationDays, durationHours, durationMinutes] = getDaysHoursMinutesFromMS(release.releaseEndDate - release.releaseStartDate);

        return {
            id: releaseId,
            rowData: {
                releases: (
                    <>
                        <DotTypography data-testid="row-release-title" variant="body1">
                            {release.releaseTitle}
                        </DotTypography>
                        <DotTypography className="release-folder-name" variant="body2">
                            {release.folderName}
                        </DotTypography>
                    </>
                ),
                status: (
                    <DotPill
                        backgroundcolor={statusBackground}
                        bordercolor={statusBorder}
                        icon={<DotIcon className={classNames('status-icon', release.releaseStatus)} iconId={calculateStatusIcon(release.releaseStatus)} />}
                        label={formatStatusText(release.releaseStatus)}
                        size="small"
                        variant="outlined"
                    />
                ),
                endDate: (
                    <DotTooltip data-testid="end-date-tooltip" placement="bottom-start" title={moment(release.releaseEndDate).format(dateFormat)}>
                        <DotTypography variant="body1">{moment(release.releaseEndDate).fromNow()}</DotTypography>
                    </DotTooltip>
                ),
                duration: (
                    <DotTypography data-testid="row-release-duration" variant="body1">
                        {formatReleaseDuration(durationDays, durationHours, durationMinutes)}
                    </DotTypography>
                ),
            },
        };
    });

    const onRowClick = (id: string) => {
        window.location.href = `#/releases/${id}`;
    };

    const mainContent = (
        <DotTable className="active-release-table" columns={tableColumns} data={releasesData} onRowClick={(_event, id) => onRowClick(id)} sortable={false} />
    );

    return (
        <div className="active-release-tile">
            <CommonActiveTile
                avatarIcon="release"
                content={mainContent}
                primaryButton={primaryButton}
                secondaryButton={secondaryButton}
                subtitle="Last updated - Most recent 5"
                title={title}
            />
        </div>
    );
};
