import { all, call, put, takeLatest } from 'redux-saga/effects';
import { SagaIterator } from 'redux-saga';
import { IHttpResponse } from 'angular';
import workflowTileSaga, {
    createReleaseErrorSaga,
    createReleaseSuccessSaga,
    initSaga,
    runWorkflowSaga,
    searchCategories,
    searchFolders,
    searchWorkflows,
    filterWorkflowTileSaga,
} from './workflow-tile.saga';
import { WorkflowPagination, workflowTile, WorkflowTileSearch } from './workflow-tile.reducer';
import { Category, Folder, Workflow } from '../../../../../../../../../../../core/xlr-ui/app/types';
import { Page } from '../../../../../../../../../../../core/xlr-ui/app/js/http/backend';
import { httpGET, httpPOST } from '../../../../../../../../../../../core/xlr-ui/app/features/common/services/http';
import getAngularService from '../../../../../../../../../../../core/xlr-ui/app/features/common/services/angular-accessor';
import { FoldersService } from '../../../../../../../../../../../core/xlr-ui/app/features/tasks/types/angular';
import { applicationManagement } from '../../../../../../../../../../xlr-application-pipelines-module/src/main/resources/web/include/applications-management/ducks/managed-application.reducer';
import { Template } from '../../../../../../../../../../xlr-application-pipelines-module/src/main/resources/web/include/applications-management/managed-application.types';

const {
    filterWorkflowTile,
    init,
    runWorkflow,
    setCategories,
    setDialogError,
    setDialogOpen,
    setFolders,
    setTotalWorkflows,
    setWorkflowTileSearch,
    setWorkflows,
} = workflowTile.actions;
const { createRelease, createReleaseError, createReleaseSuccess } = applicationManagement.actions;

describe('workflowTileSaga', () => {
    const categories: Array<Category> = [{ id: '1', title: 'Cloud', active: true }];
    const workflows: Array<Workflow> = [{ id: '1', title: 'workflow' }] as never;
    const folders: Array<Folder> = [{ id: '1', title: 'Folder' }];

    it('should yield all effects', () => {
        const gen: SagaIterator = workflowTileSaga();
        expect(gen.next().value).toStrictEqual(
            all([
                takeLatest(init, initSaga),
                takeLatest(runWorkflow, runWorkflowSaga),
                takeLatest(filterWorkflowTile, filterWorkflowTileSaga),
                takeLatest(createReleaseError, createReleaseErrorSaga),
                takeLatest(createReleaseSuccess, createReleaseSuccessSaga),
            ]),
        );
        expect(gen.next().done).toBe(true);
    });

    describe('initSaga', () => {
        it('should call proper saga', () => {
            const gen: SagaIterator = initSaga();
            const pagination: WorkflowPagination = {
                itemsPerPage: 10,
                page: 0,
            };
            expect(gen.next().value).toStrictEqual(call(searchCategories));
            expect(gen.next(categories).value).toStrictEqual(put(setCategories(categories)));
            expect(gen.next(pagination).value).toStrictEqual(call(searchWorkflows, pagination));
            expect(gen.next(workflows).value).toStrictEqual(put(setWorkflows(workflows)));
            expect(gen.next().value).toStrictEqual(call(searchFolders));
            expect(gen.next(folders).value).toStrictEqual(put(setFolders(folders)));
            expect(gen.next().done).toBe(true);
        });
    });

    describe('searchCategories', () => {
        const workflowResponse: IHttpResponse<Page<Category>> = {
            data: {
                content: { id: '1' },
            },
        } as never;

        it('should call proper sagas', () => {
            const gen: SagaIterator = searchCategories();
            expect(gen.next().value).toStrictEqual(call(httpGET, 'api/v1/categories?active=true'));
            expect(gen.next(workflowResponse)).toStrictEqual({ value: { id: '1' }, done: true });
        });
    });

    describe('searchFolders', () => {
        const foldersService: FoldersService = {
            getListOrCachedPromise: jest.fn(),
        };

        const folderResponse: IHttpResponse<Array<Folder>> = {
            data: [],
        } as never;

        it('should call proper sagas', () => {
            const gen: SagaIterator = searchFolders();
            expect(gen.next().value).toStrictEqual(call(getAngularService, 'FoldersService'));
            expect(gen.next(foldersService).value).toStrictEqual(call([foldersService, foldersService.getListOrCachedPromise], false));
            expect(gen.next(folderResponse)).toStrictEqual({ value: [], done: true });
        });
    });

    describe('searchWorkflows', () => {
        const totalElements = 10;
        const categoryResponse: IHttpResponse<Page<Workflow>> = {
            data: {
                content: { id: '1' },
                totalElements: 10,
            },
        } as never;

        const workflowTileSearch: WorkflowTileSearch = {
            author: 'John',
            categories: ['a', 'b'],
            searchInput: 'Title',
            page: 0,
            itemsPerPage: 10,
        };

        it('should call proper sagas', () => {
            const gen: SagaIterator = searchWorkflows(workflowTileSearch);
            expect(gen.next().value).toStrictEqual(put(setWorkflowTileSearch(workflowTileSearch)));
            expect(gen.next().value).toStrictEqual(call(httpPOST, 'api/v1/workflows?page=0&resultsPerPage=10', workflowTileSearch));
            expect(gen.next(categoryResponse).value).toStrictEqual(put(setTotalWorkflows(totalElements)));
            expect(gen.next()).toStrictEqual({ value: { id: '1' }, done: true });
        });
    });

    describe('runWorkflowSaga', () => {
        const templateResponse: IHttpResponse<Page<Template>> = {
            data: {
                id: '1',
            },
        } as never;

        it('should call proper sagas', () => {
            const workflow = workflows[0];
            const folderId = folders[0].id;
            const gen: SagaIterator = runWorkflowSaga({ type: 'some-type', payload: { folderId, workflow } });
            expect(gen.next().value).toStrictEqual(call(httpGET, `releases/${workflow.id}`));
            expect(gen.next(templateResponse).value).toStrictEqual(
                put(
                    createRelease({
                        template: templateResponse.data as never,
                        parentId: folderId,
                        dontRedirectOnForbidden: true,
                    }),
                ),
            );
            expect(gen.next().done).toBe(true);
        });
    });

    describe('filterWorkflowTileSaga', () => {
        it('should call proper sagas', () => {
            const workflowTileSearch: WorkflowTileSearch = {
                author: 'John',
                categories: ['a', 'b'],
                searchInput: 'Title',
                page: 0,
                itemsPerPage: 50,
            };

            const gen: SagaIterator = filterWorkflowTileSaga({ payload: workflowTileSearch, type: 'some-type' });
            expect(gen.next().value).toStrictEqual(call(searchWorkflows, workflowTileSearch));
            expect(gen.next(workflows).value).toStrictEqual(put(setWorkflows(workflows)));
            expect(gen.next().done).toBe(true);
        });
    });

    describe('createReleaseErrorSaga', () => {
        it('should call proper sagas', () => {
            const gen: SagaIterator = createReleaseErrorSaga({ payload: 'whops', type: 'unknown' });
            expect(gen.next().value).toStrictEqual(put(setDialogError('whops')));
            expect(gen.next().done).toBe(true);
        });
    });

    describe('createReleaseSuccessSaga', () => {
        it('should call proper sagas', () => {
            const gen: SagaIterator = createReleaseSuccessSaga();
            expect(gen.next().value).toStrictEqual(put(setDialogOpen(undefined)));
            expect(gen.next().done).toBe(true);
        });
    });
});
