import React, { ChangeEvent, useEffect, useMemo, useState } from 'react';
import isEmpty from 'lodash/isEmpty';
import isNil from 'lodash/isNil';
import {
    CheckboxProps,
    CssCell,
    CssGrid,
    DotAlertBanner,
    DotCheckboxGroup,
    DotDialog,
    DotIconButton,
    DotInputText,
    DotTablePagination,
    DotTypography,
    RowsPerPageOption,
} from '@digital-ai/dot-components';
// eslint-disable-next-line @typescript-eslint/ban-ts-comment
// @ts-ignore we don't have types for xl-react-components
import { XlReactWidgetTreeSelect } from 'xl-react-components';
import { calculateCellProps, groupWorkflowsByCategory, OTHER_CATEGORY } from './helper';
import { useAppDispatch, useAppSelector } from '../../../../../../../../../../../../core/xlr-ui/app/js/hooks';
import { getWorkflowTileState, WorkflowPagination, workflowTile, WorkflowTileSearch } from '../../ducks/workflow-tile.reducer';
import { WorkflowCard } from './workflow-card.component';
import { WorkflowSkeletons } from './skeletons/workflow-skeletons.component';
import { WorkflowCategoriesSkeleton } from './skeletons/workflow-categories-skeleton.component';
import { Workflow } from '../../../../../../../../../../../../core/xlr-ui/app/types';
import './workflow-catalog.component.less';

const { init, filterWorkflowTile, runWorkflow, setDialogOpen, setDialogError } = workflowTile.actions;

interface WidgetFolder {
    selectedId: string;
    title: string;
}

export const ROWS_PER_PAGE_OPTIONS = [10, 50, 100];

export const WorkflowCatalog = () => {
    const dispatch = useAppDispatch();
    const {
        categories,
        workflows,
        workflowTileSearch,
        workflowDialogOpen,
        workflowDialogError,
        totalWorkflows,
        folders,
        isLoadingWorkflows,
        isLoadingCategories,
    } = useAppSelector(getWorkflowTileState);
    const [selectedFolderId, setSelectedFolderId] = useState<string | undefined>(undefined);
    const [showErrorDetails, setShowErrorDetails] = useState(false);
    const isLoading = isLoadingWorkflows || isLoadingCategories;
    const itemsPerPage = workflowTileSearch?.itemsPerPage as RowsPerPageOption;
    const defaultPagination: WorkflowPagination = {
        page: 0,
        itemsPerPage,
    };

    const getWorkflowTileSearch = (currentTileSearch: WorkflowTileSearch) =>
        workflowTileSearch ? { ...workflowTileSearch, ...currentTileSearch } : currentTileSearch;

    const checkboxOptions: CheckboxProps[] = categories.map((category) => ({
        label: category.title,
        value: category.id,
    }));
    const workflowsByCategory = groupWorkflowsByCategory(categories, workflows);

    useEffect(() => {
        dispatch(init());
    }, []);

    useEffect(() => {
        // initialize the default folder
        if (workflowDialogOpen) {
            const workflow = workflows.find((w) => w.id === workflowDialogOpen);
            if (!workflow) return;
            setSelectedFolderId(workflow.folderId);
        }
    }, [workflowDialogOpen]);

    const handleOnRun = (workflow: Workflow) => {
        if (!selectedFolderId) return;
        dispatch(runWorkflow({ workflow, folderId: selectedFolderId }));
    };

    const handleSelectFolder = (folder: WidgetFolder) => {
        dispatch(setDialogError(undefined));
        setSelectedFolderId(folder.selectedId);
    };

    const handleOnCancel = () => {
        dispatch(setDialogOpen(undefined));
        setSelectedFolderId(undefined);
    };

    const handleCategoryChange = (categoriesSelected: Array<CheckboxProps>) => {
        dispatch(filterWorkflowTile(getWorkflowTileSearch({ categories: categoriesSelected.map((cat) => cat.label as string), ...defaultPagination })));
    };

    const handleShowDetails = () => {
        setShowErrorDetails(!showErrorDetails);
    };

    const renderWorkflow = (categoryTitle: string) => {
        const workflow = workflowsByCategory[categoryTitle];
        if (isEmpty(workflow)) return;
        return (
            <div className="workflow-category" key={categoryTitle}>
                <DotTypography className="workflow-category-title" variant="h3">
                    {categoryTitle}
                </DotTypography>
                <CssGrid>
                    {workflow.map((workflowFromCategory, index) => {
                        const props = calculateCellProps(index);
                        return (
                            <CssCell {...props} key={workflowFromCategory.id}>
                                <WorkflowCard onClick={(id) => dispatch(setDialogOpen(id))} workflow={workflowFromCategory} />
                            </CssCell>
                        );
                    })}
                </CssGrid>
            </div>
        );
    };

    const hasAllCategoriesFilter =
        isEmpty(workflowTileSearch) || isEmpty(workflowTileSearch.categories) || workflowTileSearch.categories?.length === categories.length;

    const memoizedWorkflowSkeletons = useMemo(() => (isLoading ? <WorkflowSkeletons /> : undefined), [isLoading]);

    const renderWorkflowCategories = () => {
        if (isLoading) {
            return memoizedWorkflowSkeletons;
        } else {
            return (
                <>
                    {categories
                        .filter((category) => hasAllCategoriesFilter || workflowTileSearch?.categories?.includes(category.title))
                        .map((category) => renderWorkflow(category.title))}
                    {hasAllCategoriesFilter && renderWorkflow(OTHER_CATEGORY)}
                </>
            );
        }
    };

    const renderDialog = () => {
        const workflow = workflows.find((w) => w.id === workflowDialogOpen);
        if (!workflow) return;

        return (
            <DotDialog
                cancelButtonProps={{ label: 'Cancel' }}
                className="workflow-card-folder-dialog"
                closeIconVisible={true}
                closeOnClickAway={true}
                closeOnSubmit={false}
                onCancel={handleOnCancel}
                onSubmit={() => handleOnRun(workflow)}
                open={true}
                submitButtonProps={{ label: 'Run workflow', disabled: isNil(selectedFolderId) || !!workflowDialogError }}
                title="Choose folder"
            >
                {workflowDialogError && (
                    <DotAlertBanner
                        action={<DotIconButton iconId={showErrorDetails ? 'arrow-up' : 'arrow-down'} onClick={handleShowDetails} />}
                        severity="error"
                    >
                        <DotTypography>
                            You do not have enough permissions to run a workflow in the selected folder. Please choose another folder or contact your Release
                            Administrator for further assistance.
                        </DotTypography>
                        {showErrorDetails && (
                            <DotTypography>
                                <br />
                                {workflowDialogError}
                            </DotTypography>
                        )}
                    </DotAlertBanner>
                )}
                <DotTypography>
                    Select the folder where workflow <strong>{workflow.title}</strong> will be run.
                </DotTypography>
                <DotTypography className="persistent-label" variant="subtitle2">
                    Folder name
                </DotTypography>
                <div className="xl-react-widget-tree-select-wrapper">
                    <XlReactWidgetTreeSelect clearable={false} data={folders} onModelChange={handleSelectFolder} selectedFolderId={selectedFolderId} />
                </div>
            </DotDialog>
        );
    };

    const handlePageChange = (newPage: number) => {
        dispatch(filterWorkflowTile(getWorkflowTileSearch({ page: newPage, itemsPerPage })));
    };

    const handleRowsPerPageChange = (event: ChangeEvent<HTMLTextAreaElement | HTMLInputElement>) => {
        const newItemsPerPage = Number(event.target.value) as RowsPerPageOption;
        dispatch(filterWorkflowTile(getWorkflowTileSearch({ page: defaultPagination.page, itemsPerPage: newItemsPerPage })));
    };

    return (
        <div className="workflow-catalog">
            <div className="workflow-drawer-content-left">
                <div>
                    <DotTypography variant="subtitle2">Categories</DotTypography>
                    {isLoadingCategories ? (
                        <WorkflowCategoriesSkeleton />
                    ) : (
                        <DotCheckboxGroup
                            className="categories-filter"
                            onChange={(_event, options) => handleCategoryChange(options)}
                            options={checkboxOptions}
                            showSelectAll={true}
                        />
                    )}
                    <DotInputText
                        hasDebounce={true}
                        id="authored-by"
                        label="Authored By"
                        name="authored-by"
                        onChange={(e) => dispatch(filterWorkflowTile(getWorkflowTileSearch({ author: e.target.value.toLowerCase(), ...defaultPagination })))}
                        persistentLabel={true}
                        placeholder="Start typing to filter Author"
                    />
                </div>
            </div>
            <div className="tab-content">
                <div className="workflow-drawer-header-search">
                    <DotTypography variant="h1">Workflow catalog</DotTypography>
                    <DotInputText
                        hasDebounce={true}
                        id="search"
                        label="Search workflows"
                        name="search"
                        onChange={(e) =>
                            dispatch(filterWorkflowTile(getWorkflowTileSearch({ searchInput: e.target.value.toLowerCase(), ...defaultPagination })))
                        }
                        persistentLabel={true}
                        placeholder="Start typing to filter workflows..."
                    />
                </div>
                <div className="workflow-categories">{renderWorkflowCategories()}</div>
                <DotTablePagination
                    count={totalWorkflows}
                    labelRowsPerPage="Workflows per page:"
                    onPageChange={handlePageChange}
                    onRowsPerPageChange={handleRowsPerPageChange}
                    page={workflowTileSearch?.page}
                    rowsPerPage={itemsPerPage}
                    rowsPerPageOptions={ROWS_PER_PAGE_OPTIONS}
                    typography="body1"
                />
                {renderDialog()}
            </div>
        </div>
    );
};
