import { all, call, put, takeLatest } from 'redux-saga/effects';
import { SagaIterator } from 'redux-saga';
import { IHttpResponse } from 'angular';
import workflowTileSaga, { initSaga, runWorkflowSaga, searchCategories, searchFolders, searchWorkflows } from './workflow-tile.saga';
import { workflowTile } from './workflow-tile.reducer';
import { Category, Folder, Workflow } from '../../../../../../../../../../../core/xlr-ui/app/types';
import { Page } from '../../../../../../../../../../../core/xlr-ui/app/js/http/backend';
import { httpGET, httpPOST } from '../../../../../../../../../../../core/xlr-ui/app/features/common/services/http';
import getAngularService from '../../../../../../../../../../../core/xlr-ui/app/features/common/services/angular-accessor';
import { FoldersService } from '../../../../../../../../../../../core/xlr-ui/app/features/tasks/types/angular';

const { init, setCategories, setFolders, setWorkflows, runWorkflow } = workflowTile.actions;

describe('workflowTileSaga', () => {
    const categories: Array<Category> = [{ id: '1', title: 'Cloud', active: true }];
    const workflows: Array<Workflow> = [{ id: '1', title: 'workflow' }] as never;
    const folders: Array<Folder> = [{ id: '1', title: 'Folder' }];

    it('should yield all effects', () => {
        const gen: SagaIterator = workflowTileSaga();
        expect(gen.next().value).toStrictEqual(all([takeLatest(init, initSaga), takeLatest(runWorkflow, runWorkflowSaga)]));
        expect(gen.next().done).toBe(true);
    });

    describe('initSaga', () => {
        it('should call proper saga', () => {
            const gen: SagaIterator = initSaga();
            expect(gen.next().value).toStrictEqual(call(searchCategories));
            expect(gen.next(categories).value).toStrictEqual(put(setCategories(categories)));
            expect(gen.next().value).toStrictEqual(call(searchWorkflows));
            expect(gen.next(workflows).value).toStrictEqual(put(setWorkflows(workflows)));
            expect(gen.next().value).toStrictEqual(call(searchFolders));
            expect(gen.next(folders).value).toStrictEqual(put(setFolders(folders)));
            expect(gen.next().done).toBe(true);
        });
    });

    describe('searchCategories', () => {
        const workflowResponse: IHttpResponse<Page<Category>> = {
            data: {
                content: { id: '1' },
            },
        } as never;

        it('should call proper sagas', () => {
            const gen: SagaIterator = searchCategories();
            expect(gen.next().value).toStrictEqual(call(httpGET, 'api/v1/categories'));
            expect(gen.next(workflowResponse)).toStrictEqual({ value: { id: '1' }, done: true });
        });
    });

    describe('searchFolders', () => {
        const foldersService: FoldersService = {
            getListOrCachedPromise: jest.fn(),
        };

        const folderResponse: IHttpResponse<Array<Folder>> = {
            data: [],
        } as never;

        it('should call proper sagas', () => {
            const gen: SagaIterator = searchFolders();
            expect(gen.next().value).toStrictEqual(call(getAngularService, 'FoldersService'));
            expect(gen.next(foldersService).value).toStrictEqual(call([foldersService, foldersService.getListOrCachedPromise], false));
            expect(gen.next(folderResponse)).toStrictEqual({ value: [], done: true });
        });
    });

    describe('searchWorkflows', () => {
        const categoryResponse: IHttpResponse<Page<Workflow>> = {
            data: {
                content: { id: '1' },
            },
        } as never;

        it('should call proper sagas', () => {
            const gen: SagaIterator = searchWorkflows();
            expect(gen.next().value).toStrictEqual(call(httpPOST, 'api/v1/workflows', {}));
            expect(gen.next(categoryResponse)).toStrictEqual({ value: { id: '1' }, done: true });
        });
    });
});
