import React from 'react';
import { mountWithTheme, ReactWrapper } from '../../../../../../../../../core/xlr-ui/tests/unit/testing-utils';
import * as angularAccessor from '../../../../../../../../../core/xlr-ui/app/features/common/services/angular-accessor';
import { VariablesInterpolatorFactory } from '../../../../../../../../../core/xlr-ui/app/features/tasks/types/angular';
import { TextAreaMarkdown } from '../../../../../../../../../core/xlr-ui/app/react/components/markdown/textarea-markdown.component';
import { TileButtonsRow } from './tile-buttons-row.component';
import { primaryButton, secondaryButton } from './tile-buttons-row.mock';
import { CommonCardTileComponent, CommonCardTileComponentProps, MAX_DESCRIPTION_SIZE, MAX_DESCRIPTION_SIZE_WARNING } from './common-card-tile.component';
import { CommonTileHeader } from './common-tile-header.component';

describe('CommonCardTileComponent', () => {
    const getAngularServiceSpy = jest.spyOn(angularAccessor, 'default') as unknown as jest.SpyInstance<VariablesInterpolatorFactory, [name: unknown]>;

    const description = 'my description';
    const isMarkdown = true;
    const title = 'title';
    const subtitle = 'subtitle';
    const imageLink = 'image.png';

    const defaultProps: CommonCardTileComponentProps = { description, isMarkdown, title, primaryButton };

    const mount = (props: CommonCardTileComponentProps = defaultProps) => {
        return mountWithTheme(<CommonCardTileComponent {...props} />);
    };

    const getCardHeader = (wrapper: ReactWrapper) => wrapper.find(CommonTileHeader);
    const getMarkdownDescription = (wrapper: ReactWrapper) => wrapper.find(TextAreaMarkdown);
    const getImage = (wrapper: ReactWrapper) => wrapper.find('.card-tile-content img');
    const getTileButtonsRow = (wrapper: ReactWrapper) => wrapper.find(TileButtonsRow);

    beforeEach(() => {
        getAngularServiceSpy.mockReturnValue({
            interpolateInText: jest.fn().mockReturnValue(''),
        });
    });

    describe('description', () => {
        it('should render description in markdown', () => {
            const wrapper = mount();
            const markdownProps = getMarkdownDescription(wrapper).props();
            expect(markdownProps.text).toBe(description);
            expect(markdownProps.isReadOnly).toBe(true);
            expect(markdownProps.rows).toBe(5);
        });

        it('should render description in text', () => {
            const wrapper = mount({ ...defaultProps, isMarkdown: false });
            expect(getMarkdownDescription(wrapper)).not.toExist();
            expect(wrapper).toIncludeText(description);
        });

        it('should render warning if text is too long', () => {
            const fakeDescription = {
                length: MAX_DESCRIPTION_SIZE + 1,
            };
            const wrapper = mount({ ...defaultProps, description: fakeDescription as never });
            const markdownProps = getMarkdownDescription(wrapper).props();
            expect(markdownProps.text).toBe(MAX_DESCRIPTION_SIZE_WARNING);
        });
    });

    describe('tile header', () => {
        it('should render properly', () => {
            const wrapper = mount();
            expect(getCardHeader(wrapper).props().title).toBe(title);
            expect(getCardHeader(wrapper).props().avatarIcon).toBe(undefined);
            expect(getCardHeader(wrapper).props().subtitle).toBe(undefined);
        });

        it('should render subtitle', () => {
            const wrapper = mount({ ...defaultProps, subtitle });
            expect(getCardHeader(wrapper).props().subtitle).toBe(subtitle);
        });
    });

    describe('image', () => {
        it('it should render the image if imageLink is present', () => {
            const wrapper = mount({ ...defaultProps, imageLink });
            const image = getImage(wrapper);
            expect(image).toExist();
            expect(image).toHaveProp('src', 'image.png');
        });

        it('it should not render the image if imageLink is not present', () => {
            const wrapper = mount();
            expect(getImage(wrapper)).not.toExist();
        });
    });

    describe('tile buttons row', () => {
        it('should render tile buttons row properly', () => {
            const wrapper = mount();
            const tileButtonsRowProps = getTileButtonsRow(wrapper).props();
            expect(tileButtonsRowProps.primaryButton).toBe(primaryButton);
            expect(tileButtonsRowProps.secondaryButton).toBe(undefined);
            expect(tileButtonsRowProps.alignButtonsRight).toBe(undefined);
        });

        it('should render secondary button', () => {
            const wrapper = mount({ ...defaultProps, secondaryButton });
            expect(getTileButtonsRow(wrapper).props().secondaryButton).toBe(secondaryButton);
        });
    });
});
