import React, { useEffect, useState } from 'react';
import { CheckboxProps, CssCell, CssGrid, DotCheckboxGroup, DotInputText, DotTypography } from '@digital-ai/dot-components';
import { calculateCellProps, groupWorkflowsByCategory, OTHER_CATEGORY } from './helper';
import { useAppDispatch, useAppSelector } from '../../../../../../../../../../../../core/xlr-ui/app/js/hooks';
import { getWorkflowTileState, workflowTile } from '../../ducks/workflow-tile.reducer';
import { WorkflowCard } from './workflow-card.component';
import './workflow-catalog.component.less';

const { init } = workflowTile.actions;

export const WorkflowCatalog = () => {
    const dispatch = useAppDispatch();
    const { categories, workflows } = useAppSelector(getWorkflowTileState);
    const [categoriesFilter, setCategoriesFilter] = useState<Array<string | undefined>>([]);
    const [searchFilter, setSearchFilter] = useState('');
    const [authorFilter, setAuthorFilter] = useState('');

    const checkboxOptions: CheckboxProps[] = categories.map((category) => ({
        label: category.title,
        value: category.id,
    }));
    const workflowsByCategory = groupWorkflowsByCategory(categories, workflows, searchFilter, authorFilter);

    useEffect(() => {
        dispatch(init());
    }, []);

    const handleCategoryChange = (categoriesSelected: Array<CheckboxProps>) => {
        setCategoriesFilter(categoriesSelected.map((cat) => cat.label));
    };

    const renderWorkflow = (categoryTitle: string) => {
        const workflow = workflowsByCategory[categoryTitle];
        if (!workflow || !workflow.length) return;
        return (
            <div className="workflow-category" key={categoryTitle}>
                <DotTypography className="workflow-category-title" variant="h3">
                    {categoryTitle}
                </DotTypography>
                <CssGrid>
                    {workflow.map((workflowFromCategory, index) => {
                        const props = calculateCellProps(index);
                        return (
                            <CssCell {...props} key={workflowFromCategory.id}>
                                <WorkflowCard {...workflowFromCategory} />
                            </CssCell>
                        );
                    })}
                </CssGrid>
            </div>
        );
    };

    const isAllFilterSelected = categoriesFilter.length === 0 || categoriesFilter.length === categories.length;

    return (
        <div className="workflow-catalog">
            <div className="workflow-drawer-content-left">
                <div>
                    <DotTypography variant="subtitle2">Categories</DotTypography>
                    <DotCheckboxGroup
                        className="categories-filter"
                        onChange={(_event, options) => handleCategoryChange(options)}
                        options={checkboxOptions}
                        showSelectAll={true}
                    />
                    <DotInputText
                        id="authored-by"
                        label="Authored By"
                        name="authored-by"
                        onChange={(e) => setAuthorFilter(e.target.value.toLowerCase())}
                        persistentLabel={true}
                        placeholder="Start typing to filter Author"
                    />
                </div>
            </div>
            <div className="tab-content">
                <div className="workflow-drawer-header-search">
                    <DotTypography variant="h1">Workflow catalog</DotTypography>
                    <DotInputText
                        id="search"
                        label="Search workflows"
                        name="search"
                        onChange={(e) => setSearchFilter(e.target.value.toLowerCase())}
                        persistentLabel={true}
                        placeholder="Start typing to filter workflows..."
                    />
                </div>
                <div className="workflow-categories">
                    {categories
                        .filter((category) => isAllFilterSelected || categoriesFilter.includes(category.title))
                        .map((category) => renderWorkflow(category.title))}
                    {isAllFilterSelected && renderWorkflow(OTHER_CATEGORY)}
                </div>
            </div>
        </div>
    );
};
