import React from 'react';
import moment from 'moment';
import { DotChip, DotIcon, DotIconButton, DotTooltip, DotMenu } from '@digital-ai/dot-components';
import { mountWithTheme, ReactWrapper } from '../../../../../../../../../../core/xlr-ui/tests/unit/testing-utils';
import { HomeTemplate } from './active-template-tile.component';
import { menuItemEdit, menuItemRelease, TemplateListItemEndIcon } from './template-list-item-end-icon.component';
import { MenuItem } from '../../../../../../../../../../core/xlr-ui/app/react/components/menu-item/menu-item.component';

describe('TemplateListEndIcon', () => {
    const onOptionsClick = jest.fn();
    const selected = false;
    const template: HomeTemplate = {
        releaseId: 'releaseId',
        releaseTitle: 'Test Release',
        folderName: 'test',
        folderId: 'folderId',
        folderPath: '/',
        lastEditedByUser: 1688372003150,
        releaseCount: 0,
    };
    const urlTemplateId = 'folderId-releaseId';

    const mount = () => {
        return mountWithTheme(
            <TemplateListItemEndIcon onOptionsClick={onOptionsClick} selected={selected} template={template} urlTemplateId={urlTemplateId} />,
        );
    };

    const getIconButton = (wrapper: ReactWrapper) => wrapper.find(DotIconButton);
    const getChip = (wrapper: ReactWrapper) => wrapper.find(DotChip);
    const getChipTooltip = (wrapper: ReactWrapper) => wrapper.find(DotTooltip).at(0);
    const getChipIcon = (wrapper: ReactWrapper) => wrapper.find(DotChip).find(DotIcon);
    const getMenu = (wrapper: ReactWrapper) => wrapper.find(DotMenu);
    const getReleaseMenuItem = (wrapper: ReactWrapper) => wrapper.find(MenuItem).at(0);
    const getEditMenuItem = (wrapper: ReactWrapper) => wrapper.find(MenuItem).at(1);

    it('should render with correct props', () => {
        const wrapper = mount();
        const iconButtonProps = getIconButton(wrapper).props();
        expect(iconButtonProps.iconId).toBe('options');
        expect(iconButtonProps.className).toBe('active-tile-end-icon-button');
        const chipProps = getChip(wrapper).props();
        expect(chipProps.size).toBe('small');
        expect(chipProps.children).toBe(moment(template.lastEditedByUser).fromNow());
        expect(getChipTooltip(wrapper).props().title).toBe(`Last time updated ${moment(template.lastEditedByUser).format('Do MMM YYYY')}`);
        expect(getChipIcon(wrapper).props().iconId).toBe('calendar');
        const menuProps = getMenu(wrapper).props();
        expect(menuProps.dense).toBe(true);
        expect(menuProps.disablePortal).toBe(true);
        expect(menuProps.id).toBe(template.releaseId);
        expect(menuProps.menuItems).toHaveLength(2);
        expect(menuProps.menuPlacement).toBe('left');
        expect(menuProps.open).toBe(selected);
    });

    it('should show correct menu items', () => {
        const wrapper = mount();
        getIconButton(wrapper).invoke('onClick')?.({ stopPropagation: jest.fn() } as never);

        const releaseMenuItemProps = getReleaseMenuItem(wrapper).props();
        expect(releaseMenuItemProps.iconId).toBe('play');
        expect(releaseMenuItemProps.label).toBe('Start release');
        const editMenuItemProps = getEditMenuItem(wrapper).props();
        expect(editMenuItemProps.iconId).toBe('edit');
        expect(editMenuItemProps.label).toBe('Edit template');
    });

    it('should call onOptionsClick when clicking on options icon button', () => {
        const wrapper = mount();
        getIconButton(wrapper).invoke('onClick')?.({ stopPropagation: jest.fn() } as never);
        expect(onOptionsClick).toBeCalled();
    });

    it('should redirect to proper url when clicking menu item "Start release"', () => {
        const wrapper = mount();
        getIconButton(wrapper).invoke('onClick')?.({ stopPropagation: jest.fn() } as never);

        const menu = getMenu(wrapper);
        menu.invoke('onSelect')?.({ stopPropagation: jest.fn() } as never, 'releaseId', menuItemRelease);
        expect(window.location.href).toBe(`http://localhost/#/releases/create?fromTemplateId=${urlTemplateId}`);
    });

    it('should redirect to proper url when clicking menu item "Edit"', () => {
        const wrapper = mount();
        getIconButton(wrapper).invoke('onClick')?.({ stopPropagation: jest.fn() } as never);

        const menu = getMenu(wrapper);
        menu.invoke('onSelect')?.({ stopPropagation: jest.fn() } as never, 'releaseId', menuItemEdit);
        expect(window.location.href).toBe(`http://localhost/#/templates/${urlTemplateId}`);
    });
});
