import React from 'react';
import moment from 'moment';
import { DotButton } from '@digital-ai/dot-components';
import { mockResizeObserver, mountWithStoreAndTheme, ReactWrapper } from '../../../../../../../../../../core/xlr-ui/tests/unit/testing-utils';
import { WorkflowTile, WorkflowTileToggleEnum } from './workflow-tile.component';
import { CommonCardTileComponent } from '../../CommonCardTile/common-card-tile.component';
import { ActiveWorkflowTile } from './active-workflow-tile.component';
import * as angularAccessor from '../../../../../../../../../../core/xlr-ui/app/features/common/services/angular-accessor';
import { initialState, workflow } from '../../../../../../../../../../core/xlr-ui/app/features/workflow/ducks/workflow.reducer';
import { DATE_FORMAT_DAY_FIRST } from '../../../../../../../../../../core/xlr-ui/app/js/locale/constants';

const { init } = workflow.actions;

describe('WorkflowTile', () => {
    const getAngularServiceSpy = jest.spyOn(angularAccessor, 'default') as unknown as jest.SpyInstance;

    const title = 'Workflows';
    const description = 'Start using workflows to execute small orchestrated set of tasks';
    const dispatch = jest.fn();
    const workflowTile = {
        activeTileData: {
            recent: [
                {
                    numberOfExecutions: 2,
                    categories: ['Application onboarding', 'Security'],
                    releaseId: 'myId1',
                    logoId: null,
                    releaseTitle: 'My pretty workflow',
                    folderId: 'folderId',
                    folderName: 'folder1',
                    folderPath: '/',
                },
            ],
            running: [],
        },
        mostPopular: [
            {
                allowTargetFolderOverride: false,
                author: 'someone',
                categories: ['Cloud'],
                defaultTargetFolder: 'targetFolderName',
                description: 'some description',
                executions: 5,
                folderId: 'folderId',
                folderTitle: 'folderTitle',
                id: 'workflowId',
                logo: null,
                scmTraceabilityData: {
                    commit: null,
                    remote: null,
                },
                tags: [],
                title: 'Workflow title',
            },
        ],
    };

    const defaultState = {
        workflow: { ...initialState, isLoadingWorkflowTile: false },
        profile: { dateFormat: DATE_FORMAT_DAY_FIRST },
    };

    const mount = (state = defaultState) => {
        return mountWithStoreAndTheme(<WorkflowTile description={description} title={title} />, dispatch, state);
    };

    const getCommonCardTileComponent = (wrapper: ReactWrapper) => wrapper.find(CommonCardTileComponent);
    const getActiveTileComponent = (wrapper: ReactWrapper) => wrapper.find(ActiveWorkflowTile);
    const getPrimaryButton = (wrapper: ReactWrapper) => wrapper.find(DotButton).at(0);
    const getSecondaryButton = (wrapper: ReactWrapper) => wrapper.find(DotButton).at(1);
    const getTertiaryButton = (wrapper: ReactWrapper) => wrapper.find(DotButton).at(2);

    beforeEach(() => {
        getAngularServiceSpy.mockReturnValue({
            interpolateInText: jest.fn().mockReturnValue(''),
        });

        const dateFilterWrapper = (_filter: string) => (date: string, angularJsFormat: string) => moment(date).format(angularJsFormat);
        getAngularServiceSpy.mockReturnValue(dateFilterWrapper);
    });

    beforeAll(() => {
        mockResizeObserver();
    });

    afterAll(() => {
        jest.restoreAllMocks();
    });

    it('should render properly for empty state tile (ie not active tile)', () => {
        const wrapper = mount();
        const props = getCommonCardTileComponent(wrapper).props();
        expect(props.title).toBe(title);
        expect(props.isMarkdown).toBe(true);
        expect(props.description).toBe(description);
        expect(props.imageLink).toBe('static/23.3.0-1110.100/styles/img/represent_the_stepper.svg');
        expect(props.primaryButton.buttonCaption).toBe('Create workflow');
        expect(props.primaryButton.buttonType).toBe('text');
        expect(props.secondaryButton?.buttonCaption).toBe('Open workflow catalog');
        expect(props.secondaryButton?.buttonType).toBe('outlined');
        expect(props.tertiaryButton).toBe(undefined);
    });

    it('should render properly for active tile', () => {
        const wrapper = mount({ ...defaultState, workflow: { ...defaultState.workflow, workflowTile } });
        expect(getCommonCardTileComponent(wrapper)).not.toExist();
        const props = getActiveTileComponent(wrapper).props();
        expect(props.title).toBe('Workflow executions');
        expect(props.subtitle).toBe('Account workflows in Last 30 days - Top 5');
        expect(props.primaryButton.buttonCaption).toBe('Manage');
        expect(props.primaryButton.buttonType).toBe('text');
        expect(props.secondaryButton?.buttonCaption).toBe('Create workflow');
        expect(props.secondaryButton?.buttonType).toBe('text');
        expect(props.tertiaryButton?.buttonCaption).toBe('Open workflow catalog');
        expect(props.tertiaryButton?.buttonType).toBe('outlined');
        expect(props.tableType).toBe(WorkflowTileToggleEnum.MostPopular);
    });

    it('should hide manage button if no workflows', () => {
        const wrapper = mount({ ...defaultState, workflow: { ...defaultState.workflow, workflowTile: { ...workflowTile, mostPopular: [] } } });
        const props = getActiveTileComponent(wrapper).props();
        expect(props.primaryButton.buttonCaption).toBe('Create workflow');
        expect(props.primaryButton.buttonType).toBe('text');
        expect(props.secondaryButton?.buttonCaption).toBe('Open workflow catalog');
        expect(props.secondaryButton?.buttonType).toBe('outlined');
        expect(props.tertiaryButton).toBeUndefined();
    });

    it('should render properly when Recent or Running tab is selected', () => {
        const wrapper = mount({ ...defaultState, workflow: { ...defaultState.workflow, workflowTile } });

        wrapper.find('.dot-button-toggle button').at(1).simulate('click');
        expect(getActiveTileComponent(wrapper).props().workflows).toStrictEqual(workflowTile.activeTileData.recent);

        wrapper.find('.dot-button-toggle button').at(2).simulate('click');
        expect(getActiveTileComponent(wrapper).props().workflows).toStrictEqual(workflowTile.activeTileData.running);
    });

    describe('buttons in empty state tile', () => {
        it('should handle click event for the primary button', () => {
            const wrapper = mount();
            const button = getPrimaryButton(wrapper);
            button.invoke('onClick')?.('' as never);
            expect(window.location.href).toBe('http://localhost/#/workflow-templates');
        });

        it('should handle click event for the secondary button', () => {
            const wrapper = mount();
            const button = getSecondaryButton(wrapper);
            button.invoke('onClick')?.('' as never);
            expect(dispatch).toHaveBeenCalledWith(init());
        });
    });

    describe('buttons in active state tile', () => {
        const state = { ...defaultState, workflow: { ...defaultState.workflow, workflowTile } };
        it('should handle click event for the primary button', () => {
            const wrapper = mount(state);
            const button = getPrimaryButton(wrapper);
            button.invoke('onClick')?.('' as never);
            expect(window.location.href).toBe('http://localhost/#/workflow-releases');
        });

        it('should handle click event for the secondary button', () => {
            const wrapper = mount(state);
            const button = getSecondaryButton(wrapper);
            button.invoke('onClick')?.('' as never);
            expect(window.location.href).toBe('http://localhost/#/workflow-templates');
        });

        it('should handle click event for the tertiary button', () => {
            const wrapper = mount(state);
            const button = getTertiaryButton(wrapper);
            button.invoke('onClick')?.('' as never);
            expect(dispatch).toHaveBeenCalledWith(init());
        });
    });
});
